    <?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

use Joomla\CMS\Factory;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Installer\InstallerScript;
    use Joomla\CMS\Language\Text;
    use Joomla\CMS\Table\Table;
    use Joomla\Component\Categories\Administrator\Table\CategoryTable;
    use Joomla\Database\Exception\ExecutionFailureException;

defined('_JEXEC') or die;


//the name of the class must be the name of your component + InstallerScript
//for example: com_contentInstallerScript for com_content.
class com_eventgalleryInstallerScript extends InstallerScript
{
    protected $minimumPHPVersion = '8.0.0';
    protected $minimumJoomlaVersion = '4.2.88';
	protected $initialDbVersion = null;


    /*
    * $parent is the class calling this method.
    * $type is the type of change (install, update or discover_install, not uninstall).
    * preflight runs before anything else and while the extracted files are in the uploaded temp folder.
    * If preflight returns false, Joomla will abort the update and undo everything already done.
    */
    function preflight( /** @noinspection PhpUnusedParameterInspection */$type, $parent ) {

        if ($type == 'uninstall') {
            return;
        }

        $dbVersion = $this->getDatabaseVersion();
        $this->initialDbVersion = $dbVersion;

        if ($dbVersion!= null && version_compare($dbVersion, '5.4.5_2025-02-08', 'gt')) {
            $msg = "Downgrades are not supported. Please install the same or a newer version. Current version: " . $dbVersion . '. I tried to install database version 5.4.5_2025-02-08';
            throw new \Exception($msg, 100);
        }

        $folders = array(
            //old folders will get removed by Joomla anyway
            #JPATH_ROOT . '/administrator/components/com_eventgallery/controllers',
            #JPATH_ROOT . '/administrator/components/com_eventgallery/models',
            #JPATH_ROOT . '/administrator/components/com_eventgallery/views',
            #JPATH_ROOT . '/components/com_eventgallery/controllers',
            #JPATH_ROOT . '/components/com_eventgallery/library',
            #JPATH_ROOT . '/components/com_eventgallery/models',
            #JPATH_ROOT . '/components/com_eventgallery/views',

            //new since J4 with Event Gallery 5
            JPATH_ROOT . '/administrator/components/com_eventgallery/forms',
            JPATH_ROOT . '/administrator/components/com_eventgallery/helpers',
            JPATH_ROOT . '/administrator/components/com_eventgallery/layouts',
            JPATH_ROOT . '/administrator/components/com_eventgallery/services',
            JPATH_ROOT . '/administrator/components/com_eventgallery/src',
            JPATH_ROOT . '/administrator/components/com_eventgallery/tmpl',
            // don't delete the SQL folder. This will cause trouble with the database update
            //JPATH_ROOT . '/administrator/components/com_eventgallery/sql',

            JPATH_ROOT . '/components/com_eventgallery/form',
            JPATH_ROOT . '/components/com_eventgallery/helpers',
            JPATH_ROOT . '/components/com_eventgallery/language',
            JPATH_ROOT . '/components/com_eventgallery/src',
            JPATH_ROOT . '/components/com_eventgallery/tmpl',
            JPATH_ROOT . '/components/com_eventgallery/vendor',

            // old folders which need cleanup
            JPATH_ROOT . '/components/com_eventgallery/media',
            JPATH_ROOT . '/components/com_eventgallery/tests',
            JPATH_ROOT . '/components/com_eventgallery/smarty',

            // Media Folder
            JPATH_ROOT . '/media/com_eventgallery',

            // cache folders
            JPATH_ROOT . '/cache/com_eventgallery_flickr',
            JPATH_ROOT . '/cache/com_eventgallery_template_compile',
        );

        $files = array(
            JPATH_ROOT . '/language/en-GB/en-GB.com_eventgallery.ini',
            JPATH_ROOT . '/language/de-DE/de-DE.com_eventgallery.ini',
            JPATH_ROOT . '/administrator/language/en-GB/en-GB.com_eventgallery.ini',
            JPATH_ROOT . '/administrator/language/en-GB/en-GB.com_eventgallery.sys.ini',
            JPATH_ROOT . '/administrator/components/com_eventgallery/sql/updates/mysql/3.10.16_2019-02-27.sql',
            JPATH_ROOT . '/administrator/components/com_eventgallery/sql/updates/mysql/all.sql',
            // Joomla 4 migration with Event Gallery 5
            JPATH_ROOT . '/components/com_eventgallery/eventgallery.php',
            JPATH_ROOT . '/components/com_eventgallery/controller.php',
            JPATH_ROOT . '/administrator/components/com_eventgallery/eventgallery.php',
            JPATH_ROOT . '/administrator/components/com_eventgallery/controller.php',

            //old CLI Scripts
            JPATH_ROOT . '/cli/eventgallery-sync.php',
            JPATH_ROOT . '/cli/eventgallery-s3-thumbnails.php',
            JPATH_ROOT . '/cli/eventgallery-local-thumbnails.php',
        );


        foreach($folders as $folder) {
            if (is_dir($folder)) {
                Folder::delete($folder);
            }
        }

        foreach($files as $file) {
            if (is_dir($file)) {
                Folder::delete($file);
            }
        }

    }


    function uninstall( /** @noinspection PhpUnusedParameterInspection */$parent ) {

	}

    function postflight( /** @noinspection PhpUnusedParameterInspection */$type, $parent )
    {
        if($type == 'uninstall') {
            return;
        }

        $db = Factory::getDbo();

        if ($type == 'update') {
            $this->updateDatabase(false);
        } else {
            $this->updateDatabase(true);
        }

        $this->handlePlugins($db);
        $this->updateContentTypes();
        $this->createDefaultCategory();
        $this->removeOldColumns();
        $this->moveLocalImageCacheFolder();

        $this->conditionalInstallDashboard('com-eventgallery-overview', 'eventgallery');

        $this->showReleaseNotes();
    }

    /**
     * Loads the ID of the extension from the database
     *
     * @return mixed
     */
    private function getExtensionId() {
        $db = Factory::getDbo();

        $query = $db->getQuery(true);
        $query->select('extension_id')
            ->from('#__extensions')
            ->where($db->qn('element').' = '.$db->q('com_eventgallery'). ' AND type='. $db->q('component'));
        $db->setQuery($query);
        $eid = $db->loadResult();

        return $eid;
    }

    private function getDatabaseVersion() {
        // Get the extension ID
        $db = Factory::getDbo();


        $eid = $this->getExtensionId();

        if ($eid != null) {
            // Get the schema version
            $query = $db->getQuery(true);
            $query->select('version_id')
                ->from('#__schemas')
                ->where('extension_id = ' . $db->quote($eid));
            $db->setQuery($query);
            $version = $db->loadResult();

            return $version;
        }

        return null;
    }

    /**
     * create the default category
     */
    private function createDefaultCategory()
    {
        /**
         * @var CategoryTable $category
         */
        $category = Table::getInstance('Category', 'Joomla\\CMS\\Table\\');

        /*
         * there are categories with different aliases. We need to check both.
         */
        $category1Loaded = $category->load(array('extension' => 'com_eventgallery', 'alias' => 'uncategorised'), true);
        $category2Loaded = $category->load(array('extension' => 'com_eventgallery', 'alias' => 'uncategorized'), true);

        if (!$category1Loaded && !$category2Loaded)
        {
            $category->extension        = 'com_eventgallery';
            $category->title            = 'Uncategorised';
            $category->alias            = 'uncategorized';
            $category->description      = '';
            $category->published        = 1;
            $category->access           = 1;
            $category->params           = '{"category_layout":"","image":""}';
            $category->metadata         = '{"author":"","robots":""}';
            $category->metadesc         = '';
            $category->metakey          = '';
            $category->language         = '*';
            $category->checked_out_time = Factory::getDbo()->getNullDate();
            $category->version          = 1;
            $category->hits             = 0;
            $category->modified_user_id = 0;
            $category->checked_out      = 0;

            // Set the location in the tree
            $category->setLocation(1, 'last-child');

            // Check to make sure our data is valid
            if (!$category->check())
            {
                Factory::getApplication()->enqueueMessage(Text::sprintf('COM_EVENTGALLERY_ERROR_INSTALL_CATEGORY', $category->getError()));

                return;
            }

            // Now store the category
            if (!$category->store(true))
            {
                Factory::getApplication()->enqueueMessage(Text::sprintf('COM_EVENTGALLERY_ERROR_INSTALL_CATEGORY', $category->getError()));

                return;
            }

            // Build the path for our category
            $category->rebuildPath($category->id);
        }
    }

    /**
     * Performs the database update operations.
     *
     * @param $fixSchemaOnly
     * @return bool
     */
    private function updateDatabase($fixSchemaOnly) {
        $db = Factory::getDbo();

        // Get the folder from the database name
        $sqlFolder = $db->name;

        if ($sqlFolder == 'mysqli' || $sqlFolder == 'pdomysql')
        {
            $sqlFolder = 'mysql';
        }
        elseif ($sqlFolder == 'sqlsrv')
        {
            $sqlFolder = 'sqlazure';
        }

        $folder = JPATH_ADMINISTRATOR . '/components/com_eventgallery/sql/updates/';

        $files = Folder::files(
            $folder . '/' . $sqlFolder, '\.sql$', 1, true, array('.svn', 'CVS', '.DS_Store', '__MACOSX'), array('^\..*', '.*~'), true
        );

        $currentVersion = $this->getSchemaVersion();



        if (!$changeSet = $this->getItems())
        {
            return false;
        }

        if (!$fixSchemaOnly) {
            $targetVersion = $changeSet->getSchema();

            $updateQueries = $this->getUpdateQueries($files);

            if (strcmp($currentVersion, $targetVersion) !== 0) {
                Factory::getApplication()->enqueueMessage("Event Gallery - Updated the database schema from $currentVersion to $targetVersion. This version does not need to match the version of Event Gallery since not every update contains database changes");
            }

            foreach ($updateQueries as $obj) {
                $changeItem = \Joomla\CMS\Schema\ChangeItem::getInstance($db, $obj->file, $obj->updateQuery);


                $version = basename($obj->file, '.sql');
                if (version_compare($currentVersion, $version) == -1) {
                    echo "<pre>";
                    echo $obj->file;
                    echo "\n\n";
                    echo $changeItem->updateQuery;
                    $query = $changeItem->db->convertUtf8mb4QueryToUtf8($changeItem->updateQuery);

                    $changeItem->db->setQuery($query);
                    try {
                        $changeItem->db->execute();
                    } catch (\Exception $e) {
                        Factory::getApplication()->enqueueMessage($e->getMessage(), 'warning');
                    }

                    echo "</pre>";
                }
            }
        }

        $this->fixSchemaVersion($changeSet);

        return true;
    }

    private function removeOldColumns() {
        $db = \Joomla\CMS\Factory::getDbo();

        // REMOVE allowcomments from #__eventgallery_file
        $columns = $db->getTableColumns('#__eventgallery_file');
        if (array_key_exists('allowcomments', $columns)) {
            $db->setQuery('ALTER TABLE ' . $db->quoteName($db->replacePrefix('#__eventgallery_file')) . ' DROP COLUMN ' . $db->quoteName('allowcomments'));
            $db->execute();
            Factory::getApplication()->enqueueMessage('Removed obsolete column allowcomments in #__eventgallery_file');
        }
    }

    /**
     * Gets the changeset object.
     *
     * @return  \Joomla\CMS\Schema\ChangeSet
     */
    public function getItems()
    {

        $db = Factory::getDbo();

        $folder = JPATH_ADMINISTRATOR . '/components/com_eventgallery/sql/updates/';

        try
        {
            $changeSet = \Joomla\CMS\Schema\ChangeSet::getInstance($db, $folder);
        }
        catch (\RuntimeException $e)
        {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'warning');

            return false;
        }
        return $changeSet;
    }

    /**
     * Get version from #__schemas table.
     *
     * @return  mixed  the return value from the query, or null if the query fails.
     *
     * @throws \Exception
     */
    public function getSchemaVersion()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('version_id')
            ->from($db->quoteName('#__schemas'))
            ->where('extension_id = ' . $db->quote($this->getExtensionId()));
        $db->setQuery($query);
        $result = $db->loadResult();

        return $result;
    }

    /**
     * Fix schema version if wrong.
     *
     * @param   \Joomla\CMS\Schema\ChangeSet  $changeSet  Schema change set.
     *
     * @return   mixed  string schema version if success, false if fail.
     */
    public function fixSchemaVersion($changeSet)
    {
        // Get correct schema version -- last file in array.
        $schema = $changeSet->getSchema();
        $extensionid = $this->getExtensionId();

        // Check value. If ok, don't do update.
        if ($schema == $this->getSchemaVersion())
        {
            return $schema;
        }

        // Delete old row.
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->delete($db->quoteName('#__schemas'))
            ->where($db->quoteName('extension_id') . ' = ' . $db->quote($extensionid));
        $db->setQuery($query);
        $db->execute();

        // Add new row.
        $query->clear()
            ->insert($db->quoteName('#__schemas'))
            ->columns($db->quoteName('extension_id') . ',' . $db->quoteName('version_id'))
            ->values($db->quote($extensionid) . ', ' . $db->quote($schema));
        $db->setQuery($query);

        try
        {
            $db->execute();
        }
        catch (ExecutionFailureException $e)
        {
            return false;
        }

        return $schema;
    }

    /**
     * Get array of SQL queries
     *
     * @param   array  $sqlfiles  Array of .sql update filenames.
     *
     * @return  array  Array of \stdClass objects where:
     *                    file=filename,
     *                    update_query = text of SQL update query
     *
     * @since   2.5
     */
    private function getUpdateQueries(array $sqlfiles)
    {
        // Hold results as array of objects
        $result = array();

        foreach ($sqlfiles as $file)
        {
            $buffer = file_get_contents($file);

            // Create an array of queries from the sql file
            $queries = \Joomla\Database\DatabaseDriver::splitSql($buffer);

            foreach ($queries as $query)
            {
                $fileQueries = new \stdClass;
                $fileQueries->file = $file;
                $fileQueries->updateQuery = $query;
                $result[] = $fileQueries;
            }
        }

        return $result;
    }

    private function moveLocalImageCacheFolder() {
        $oldCacheFolder = JPATH_ROOT . '/cache/com_eventgallery_images';
        $newCacheFolder = JPATH_ROOT . '/images/eventgallery_generated';

        if (is_dir($oldCacheFolder) && !is_dir($newCacheFolder)) {
            $result = rename( $oldCacheFolder,  $newCacheFolder);
            if ($result) {
                $msg = "New image cache for local images! Moved the image cache folder from /cache/com_eventgallery_images to /images/eventgallery_generated. This is the new folder for generated thumbnails to keep them even if an update of Joomla wipes the cache.";
            } else {
                $msg = "New image cache for local images! Unable to move /cache/com_eventgallery_images to /images/eventgallery_generated. Please check manually to make sure, your folder for generated thumbnails is setup correctly.";
            }

            Factory::getApplication()->enqueueMessage($msg);

        }

    }

    private function updateContentTypes() {
        $this->updateContentTypeCategory();
        $this->updateContentTypeEvent();
    }

    private function updateContentTypeCategory() {
        $contentType = new ContentType();
        $contentType->type_alias = 'com_eventgallery.category';
        $contentType->type_title = 'Eventgallery Category';
        $contentType->table = '{"special":{"dbtable":"#__categories","key":"id","type":"Category","prefix":"JTable","config":"array()"},"common":{"dbtable":"#__ucm_content","key":"ucm_id","type":"Corecontent","prefix":"JTable","config":"array()"}}';
        $contentType->rules = '';
        $contentType->field_mappings = '{"common":{"core_content_item_id":"id","core_title":"title","core_state":"published","core_alias":"alias","core_created_time":"created_time","core_modified_time":"modified_time","core_body":"description", "core_hits":"hits","core_publish_up":"null","core_publish_down":"null","core_access":"access", "core_params":"params", "core_featured":"null", "core_metadata":"metadata", "core_language":"language", "core_images":"null", "core_urls":"null", "core_version":"version", "core_ordering":"null", "core_metakey":"metakey", "core_metadesc":"metadesc", "core_catid":"parent_id", "core_xreference":"null", "asset_id":"asset_id"}, "special": {"parent_id":"parent_id","lft":"lft","rgt":"rgt","level":"level","path":"path","extension":"extension","note":"note"}}';
        $contentType->router = '\Svenbluege\Component\Eventgallery\Site\Library\Helper\Route::getCategoryRoute';
        $contentType->content_history_options = '';
        $this->updateContentType($contentType);
    }

    private function updateContentTypeEvent() {
        $contentType = new ContentType();
        $contentType->type_alias = 'com_eventgallery.event';
        $contentType->type_title = 'Eventgallery Event';
        $contentType->table = '{"special":{"dbtable":"#__eventgallery_folder","key":"folder","type":"Folder","prefix":"EventgalleryTable","config":"array()"},"common":{"dbtable":"#__ucm_content","key":"ucm_id","type":"Corecontent","prefix":"JTable","config":"array()"}}';
        $contentType->rules = '';
        $contentType->field_mappings = '{"common":{"core_content_item_id":"id","core_title":"description","core_state":"published","core_alias":"null","core_created_time":"created_time","core_modified_time":"modified_time","core_body":"text", "core_hits":"hits","core_publish_up":"null","core_publish_down":"null","core_access":"access", "core_params":"params", "core_featured":"null", "core_metadata":"metadata", "core_language":"null", "core_images":"null", "core_urls":"null", "core_version":"version", "core_ordering":"null", "core_metakey":"metakey", "core_metadesc":"metadesc", "core_catid":"catid", "core_xreference":"null", "asset_id":"asset_id"}, "special": {"parent_id":"parent_id","lft":"lft","rgt":"rgt","level":"level","path":"path","extension":"extension","note":"note"}}';
        $contentType->router = '\Svenbluege\Component\Eventgallery\Site\Library\Helper\Route::getEventRoute';
        $contentType->content_history_options = '';
        $this->updateContentType($contentType);
    }

    /**
    * @param ContentType $contentType
    * @return void
     */
    private function updateContentType($contentType) {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select($db->quoteName('type_id'))
            ->from($db->quoteName('#__content_types'))
            ->where($db->quoteName('type_alias') . ' = ' . $db->quote($contentType->type_alias));
        $db->setQuery($query);
        $db->execute();

        if ($db->getNumRows() == 0) {
            $db->insertObject('#__content_types', $contentType);
        } else {
            $db->updateObject('#__content_types', $contentType, 'type_alias');
        }
    }

    /**
    * @param \Joomla\Database\DatabaseDriver|null $db
    */
    private function handlePlugins($db) {
        $plugins = array(
                array('system', 'picasaupdater'),
                array('installer', 'eventgallery')
        );

        foreach($plugins as $pluginData) {

            // Let's get the information of the update plugin
            $query = $db->getQuery(true)
                ->select('*')
                ->from($db->qn('#__extensions'))
                ->where($db->qn('folder').' = '.$db->quote($pluginData[0]))
                ->where($db->qn('element').' = '.$db->quote($pluginData[1]))
                ->where($db->qn('type').' = '.$db->quote('plugin'));
            $db->setQuery($query);
            $plugin = $db->loadObject();

            // If it's missing or enabledthere's nothing else to do
            if (!is_object($plugin) || $plugin->enabled)
            {
                continue;
            }


            // Otherwise, try to enable it
            $pluginObject = (object)array(
                'extension_id'  => $plugin->extension_id,
                'enabled'       => 1
            );

            try
            {
                $db->updateObject('#__extensions', $pluginObject, 'extension_id');
            }
            catch (\Exception $e)
            {
            }
        }
    }
    private function conditionalInstallDashboard(string $dashboard, string $preset): void
    {
        $position = 'cpanel-' . $dashboard;

        /** @var \Joomla\Database\DatabaseDriver $db */
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true)
          ->select('COUNT(*)')
          ->from($db->quoteName('#__modules'))
          ->where([
            $db->quoteName('module') . ' = ' . $db->quote('mod_submenu'),
            $db->quoteName('client_id') . ' = ' . $db->quote(1),
            $db->quoteName('position') . ' = :position',
          ])
          ->bind(':position', $position);

        $modules = $db->setQuery($query)->loadResult() ?: 0;

        if ($modules == 0) {
            $this->addDashboardMenu($dashboard, $preset);
        }
    }

    private function showReleaseNotes() {
        $msg = 'Event Gallery 5.4.10 installed successfully. Find out what is new by checking the <b>release notes</b>: <a target="_blank" href="https://www.svenbluege.de/joomla-event-gallery/event-gallery-manual/releases">Event Gallery - Release Notes</a>.';
        Factory::getApplication()->enqueueMessage($msg, \Joomla\CMS\Application\CMSApplicationInterface::MSG_NOTICE);
    }
}

class ContentType {
    public $type_alias;
    public $type_title;
    public $table;
    public $rules;
    public $field_mappings;
    public $router;
    public $content_history_options;
}

