<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Administrator\CliCommand;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Svenbluege\Component\Eventgallery\Administrator\Table\FolderTable;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FileFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FolderFactory;
use Svenbluege\Component\Eventgallery\Site\Library\File\File;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\AddResult;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\Folder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\GooglePhotosFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\LocalFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\S3Folder;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\FolderManager;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

defined('_JEXEC') or die;

class MigrateGooglePhotos extends \Joomla\Console\Command\AbstractCommand
{
    protected static $defaultName = 'eventgallery:migrate-google-photos';

    private SymfonyStyle $symfonyStyle;
    public function doExecute(InputInterface $input, OutputInterface $output): int
    {
        $this->symfonyStyle = new SymfonyStyle($input, $output);

        $dryRun = $input->getOption('dryrun') == 'false' ? false : true;

        $targetStorage = $input->getOption('target') == 's3' ? 's3' : 'local';

        define('JPATH_COMPONENT_ADMINISTRATOR', JPATH_ADMINISTRATOR . '/components/com_eventgallery');
        define('JPATH_COMPONENT_SITE', JPATH_SITE . '/components/com_eventgallery');

        $language = Factory::getLanguage();
        $language->load('com_eventgallery' , JPATH_COMPONENT_ADMINISTRATOR, $language->getTag(), true);

        $this->symfonyStyle->title("Migrating Google Photos to $targetStorage storage.");

        $folderFactory = FolderFactory::getInstance();
        $folders = $folderFactory->getAllFolders();

        foreach($folders as $folder) {
            if ($folder->getFolderType()->getId() != GooglePhotosFolder::ID) {
                continue;
            }
            $this->symfonyStyle->write("Migrating {$folder->getDisplayName()} to $targetStorage storage.\n");
            $fileCount = $folder->getFileCount();
            $this->symfonyStyle->write("    Number of files to copy: $fileCount\n");

            if ($dryRun) {
                continue;
            }

            $this->migrateFolder($folder, $targetStorage);
        }


        $this->symfonyStyle->write("\n\n");
        if ($dryRun) {
            $this->symfonyStyle->warning("Dry run finished. No files were copied.\n");
        } else {
            $this->symfonyStyle->success("Migration finished.");
        }


        return 0;
    }

    protected function configure(): void
    {
        $this->addOption('dryrun', 'd', \Symfony\Component\Console\Input\InputArgument::OPTIONAL, '', true);
        $this->addOption('target', 't', \Symfony\Component\Console\Input\InputArgument::OPTIONAL, '', 'local');
        $this->setDescription('Sync filesystem with database');
        $this->setHelp(
            <<<EOF
Event Gallery - Migrate Google Photos
#######################

This script can migrate your Google Photos to your local/S3 files.

        <info>--dryrun=[true|false]</info> or <info>-d=true</info> 
            Use this to perform the migration. By default this command will do a dry run. 
            
            Default: true
            
        <info>--target=[s3|local]</info> or <info>-t=local</info> 
            Select the target for your migration. Can be local file storage or AWS S3 storage. 
            
            Default: local

        Example: php joomla.php eventgallery:migrate-google-photos --dryrun=false --target=local

EOF
        );
    }

    private function migrateFolder(Folder $folder, $targetStorage) {
        $folderFactory = FolderFactory::getInstance();
        /**
         * @var FolderTable $table
         */
        $table = Factory::getApplication()->bootComponent('com_eventgallery')->getMVCFactory()->createTable('Folder', 'Administrator');
        $table->load($folder->getId());

        $newFolderName = str_replace(' ', '_', \Joomla\Filesystem\File::makeSafe($table->googlephotostitle . '_m'));

        $table->folder = $newFolderName;
        if ($targetStorage == 's3') {
            $table->foldertypeid = S3Folder::ID;
        } else {
            $table->foldertypeid = LocalFolder::ID;
        }
        $table->id = null;
        $success = $table->store();

        if (!$success) {
            $this->symfonyStyle->warning("    Can't create new Event for the folder name $newFolderName . The Event might already exist or there was another error creating it.\n");
        } else {
            $this->symfonyStyle->write("    Create the Event with the folder name $newFolderName \n");
        }

        $newFolder = $folderFactory->getFolder($newFolderName);

        if (!empty($newFolder)) {
            if ($targetStorage == 's3' && $newFolder->getFolderType()->getId() != S3Folder::ID) {
                $this->symfonyStyle->error("    Can't migrate folder to S3 storage because the folder is already in local storage. \n");
                return;
            }
            if ($targetStorage == 'local' && $newFolder->getFolderType()->getId() != LocalFolder::ID) {
                $this->symfonyStyle->error("    Can't migrate folder to local storage because the folder is already in S3 storage. \n");
                return;
            }

            $this->symfonyStyle->write("    I will now copy files from {$folder->getFolderName()} to {$newFolder->getFolderName()} \n");
            FolderManager::copyFiles($folder, $newFolder, $this->symfonyStyle);
            $this->symfonyStyle->write("    " . Text::_('COM_EVENTGALLERY_EVENT_MIGRATE_TO_LOCAL_COPY_SUCCESS') ." \n");
        }

    }
}
