<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Administrator\Field;

use Joomla\CMS\Form\FormField;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Factory;

class BreakpointeditorField extends FormField
{
    /**
     * The form field type.
     *
     * @var    string
     */
    protected $type = 'breakpointeditor';

    /**
     * Predefined breakpoint widths
     *
     * @var    array
     */
    protected $predefinedWidths = [1, 450, 900, 1200, 1800, 2400];

    /**
     * Method to get the field input markup.
     *
     * @return  string  The field input markup.
     */
    protected function getInput()
    {
        $wa = Factory::getApplication()->getDocument()->getWebAssetManager();
        
        // Get the current value
        $value = $this->value;
        if (empty($value)) {
            $value = $this->default;
        }

        // Decode JSON if it's a string
        if (is_string($value)) {
            $value = json_decode($value, true);
        }

        // Ensure we have a valid structure and sort by width
        if (!isset($value['breakpoints']) || !is_array($value['breakpoints'])) {
            $value = [
                'breakpoints' => [
                ]
            ];
        } else {
            // Sort breakpoints by width
            usort($value['breakpoints'], function($a, $b) {
                return $a['width'] - $b['width'];
            });
        }

        // Start building the HTML
        $html = [];
        $html[] = '<div class="breakpoint-editor" id="breakpoint-editor_'.$this->id.'">';
        
        // Table for existing breakpoints
        $html[] = '<table class="table breakpoint-list">';
        $html[] = '<thead>';
        $html[] = '<tr>';
        $html[] = '<th>' . Text::_('COM_EVENTGALLERY_OPTIONS_BREAKPOINT_EDITOR_WIDTH') . '</th>';
        $html[] = '<th>' . Text::_('COM_EVENTGALLERY_OPTIONS_BREAKPOINT_EDITOR_ITEMS_PER_ROW') . '</th>';
        $html[] = '<th></th>';
        $html[] = '</tr>';
        $html[] = '</thead>';
        $html[] = '<tbody>';

        // Add breakpoints
        foreach ($value['breakpoints'] as $index => $breakpoint) {
            $html[] = '<tr class="breakpoint-row">';
            $html[] = '<td>' . $breakpoint['width'] . '</td>';
            $html[] = '<td>' . $breakpoint['itemsPerRow'] . '</td>';
            $html[] = '<td><button type="button" class="btn btn-danger btn-sm remove-breakpoint"><span class="icon-delete" aria-hidden="true"></span></button></td>';
            $html[] = '</tr>';
        }
        
        $html[] = '</tbody>';
        $html[] = '</table>';

        // Add new breakpoint form
        $html[] = '<div class="add-breakpoint-form">';
        $html[] = '<div class="input-group">';
        
        // Width input with datalist
        $html[] = '<input type="number" list="breakpoint-widths" class="form-control new-breakpoint-width" placeholder="' . 
            Text::_('COM_EVENTGALLERY_OPTIONS_BREAKPOINT_EDITOR_WIDTH') . '" min="0" step="1">';
        $html[] = '<datalist id="breakpoint-widths">';
        foreach ($this->predefinedWidths as $width) {
            $html[] = '<option value="' . $width . '">';
        }
        $html[] = '</datalist>';

        $html[] = '<input type="number" list="items-per-row-values" class="form-control new-breakpoint-items" placeholder="' .
            Text::_('COM_EVENTGALLERY_OPTIONS_BREAKPOINT_EDITOR_ITEMS_PER_ROW') . '" min="1" step="1">';
        $html[] = '<datalist id="items-per-row-values">';
        for ($i = 1; $i<7; $i++) {
            $html[] = '<option value="' . $i . '">';
        }
        $html[] = '</datalist>';
        $html[] = '<button type="button" class="btn btn-success add-breakpoint"><span class="icon-plus" aria-hidden="true"></span></button>';
        $html[] = '</div>';
        $html[] = '</div>';

        $html[] = '<input type="hidden" name="' . $this->name . '" value="' . htmlspecialchars(json_encode($value)) . '">';
        $html[] = '</div>';

        // Add JavaScript
        $html[] = '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const editor = document.getElementById("breakpoint-editor_'.$this->id.'");
                const list = editor.querySelector(".breakpoint-list tbody");
                const input = editor.querySelector("input[type=hidden]");
                const newWidthInput = editor.querySelector(".new-breakpoint-width");
                const newItemsInput = editor.querySelector(".new-breakpoint-items");

                function updateValue() {
                    const breakpoints = [];
                    list.querySelectorAll(".breakpoint-row").forEach(row => {
                        breakpoints.push({
                            width: parseInt(row.cells[0].textContent),
                            itemsPerRow: parseInt(row.cells[1].textContent)
                        });
                    });
                    
                    // Sort breakpoints by width
                    breakpoints.sort((a, b) => a.width - b.width);
                    
                    // Update the hidden input
                    input.value = JSON.stringify({breakpoints: breakpoints});
                    
                    // Reorder DOM elements
                    const rows = Array.from(list.querySelectorAll(".breakpoint-row"));
                    rows.sort((a, b) => {
                        const widthA = parseInt(a.cells[0].textContent);
                        const widthB = parseInt(b.cells[0].textContent);
                        return widthA - widthB;
                    });
                    
                    // Clear and re-append sorted rows
                    list.innerHTML = "";
                    rows.forEach(row => list.appendChild(row));
                }

                editor.addEventListener("click", function(event) {
                    if (event.target.closest(".remove-breakpoint")) {
                        event.target.closest(".breakpoint-row").remove();
                        updateValue();
                    } else if (event.target.closest(".add-breakpoint")) {
                        const width = parseInt(newWidthInput.value);
                        const items = parseInt(newItemsInput.value);
                        
                        if (!width || !items) {
                            return;
                        }

                        const row = document.createElement("tr");
                        row.className = "breakpoint-row";
                        row.innerHTML = `
                            <td>${width}</td>
                            <td>${items}</td>
                            <td><button type="button" class="btn btn-danger btn-sm remove-breakpoint"><span class="icon-delete" aria-hidden="true"></span></button></td>
                        `;
                        
                        list.appendChild(row);
                        updateValue();
                        
                        // Clear inputs
                        newWidthInput.value = "";
                        newItemsInput.value = "";
                    }
                });

                // Add keyboard support for the form
                editor.querySelector(".add-breakpoint-form").addEventListener("keypress", function(event) {
                    if (event.key === "Enter") {
                        event.preventDefault();
                        editor.querySelector(".add-breakpoint").click();
                    }
                });
            });
        </script>';

        return implode("\n", $html);
    }
} 