<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Administrator\Table;
use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;
use Svenbluege\Component\Eventgallery\Site\Library\Common\LogWorkaround;
use Svenbluege\Component\Eventgallery\Site\Library\Database\Localizablestring;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\OrderStatusFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\EmailTemplateManager;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\OrderManager;
use Svenbluege\Component\Eventgallery\Site\Library\Order;
use Svenbluege\Component\Eventgallery\Site\Library\OrderStatus;

defined('_JEXEC') or die('Restricted access');

class OrderTable extends Table
{

    /** @var OrderTable Caches the row data on load for future reference */
    private $_selfCache = null;

    public $id;
    public $documentno;
    public $userid;
    public $language;
    public $email;
    public $phone;
    public $firstname;
    public $lastname;
    public $statusid;
    public $subtotal;
    public $subtotalcurrency;
    public $total;
    public $totalcurrency;
    public $surchargeid;
    public $paymentmethodid;
    public $shippingmethodid;
    public $billingaddressid;
    public $shippingaddressid;
    public $message;
    public $modified;
    public $created;
    public $version;
    public $token;
    public $newsletter_optin;

    public $orderstatusid;
    public $paymentstatusid;
    public $shippingstatusid;

    public $surchargetotal;
    public $surchargetotalcurrency;

    public $paymenttotal;
    public $paymenttotalcurrency;

    public $shippingtotal;
    public $shippingtotalcurrency;


    /**
     * Constructor
     * @param DatabaseDriver $db
     */

	function __construct( &$db ) {
		parent::__construct('#__eventgallery_order', 'id', $db);

        (new LogWorkaround())->registerLogger('com_eventgallery_formatted_text_logger', \Svenbluege\Component\Eventgallery\Site\Library\Common\FormattedTextLogger::class, true);
        Log::addLogger(
            array(
                'text_file' => 'com_eventgallery_order.log.php',
                'logger' => 'com_eventgallery_formatted_text_logger'
            ),
            Log::ALL,
            'com_eventgallery_order'
        );

    }

    public function store( $updateNulls=false )
	{
        $this->modified = date("Y-m-d H:i:s");
		if(!$this->onBeforeStore($updateNulls)) return false;
		$result = parent::store($updateNulls);
		if($result) {
			$result = $this->onAfterStore();
		}
		return $result;
	}

	public function load( $keys=null, $reset=true )
	{
		$result = parent::load($keys, $reset);
		$this->onAfterLoad($result);
		return $result;
	}

	/**
	 * Method to reset class properties to the defaults set in the class
	 * definition. It will ignore the primary key as well as any private class
	 * properties.
	 */
	public function reset()
	{
		parent::reset();

		if(!$this->onAfterReset()) return false;

        return true;
	}


    /**
     * Caches the loaded data so that we can check them for modifications upon
     * saving the row.
     * @param $result
     * @return bool
     */
    public function onAfterLoad(&$result)
    {
        $this->_selfCache = $result ? clone $this : null;
        return true;
    }

    /**
     * Resets the cache when the table is reset
     * @return bool
     */
    public function onAfterReset()
    {
        $this->_selfCache = null;
        return true;
    }

    /**
     * @param bool $updateNulls
     * @throws \Exception
     * @returns bool
     */
    protected function onBeforeStore(/** @noinspection PhpUnusedParameterInspection */$updateNulls = false) {

        if (!isset($this->_selfCache)) {
            return true;
        }
        // check the version flags to make sure we don't overwrite newer rows with older data.

        if (($this->_selfCache->version > $this->version)) {
            throw new \Exception(\Joomla\CMS\Language\Text::sprintf('COM_EVENTGALLERY_ORDER_SAVE_FAILED_VERSIONCONFLICT', $this->_selfCache->version, $this->version));
        } else {
            $this->version++;
        }


        return true;
    }

    /**
     * Automatically run some actions after a subscription row is saved
     */
    protected function onAfterStore()
    {

         /**
         * @var OrderManager $orderMgr
         */
        $orderMgr = OrderManager::getInstance();
        $order = $orderMgr->getOrderById($this->id);
        $config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();


        if ($order == null) {
            return true;
        }

        if ($this->_selfCache == null) {
            return true;
        }

        // trigger the method functions if a status changed
        if ($this->_selfCache->paymentstatusid != $this->paymentstatusid) {
            if (null!=$order->getPaymentMethod())   {$order->getPaymentMethod()->onPaymentStatusChange($order);}
            if (null!=$order->getShippingMethod())  {$order->getShippingMethod()->onPaymentStatusChange($order);}
            if (null!=$order->getSurcharge())       {$order->getSurcharge()->onPaymentStatusChange($order);}
        }

        if ($this->_selfCache->shippingstatusid != $this->shippingstatusid) {
            if (null!=$order->getPaymentMethod())   {$order->getPaymentMethod()->onShippingStatusChange($order);}
            if (null!=$order->getShippingMethod())  {$order->getShippingMethod()->onShippingStatusChange($order);}
            if (null!=$order->getSurcharge())       {$order->getSurcharge()->onShippingStatusChange($order);}
        }

        if ($this->_selfCache->orderstatusid != $this->orderstatusid) {
            if (null!=$order->getPaymentMethod())   {$order->getPaymentMethod()->onOrderStatusChange($order);}
            if (null!=$order->getShippingMethod())  {$order->getShippingMethod()->onOrderStatusChange($order);}
            if (null!=$order->getSurcharge())       {$order->getSurcharge()->onOrderStatusChange($order);}
        }


        $result = true;

        $disclaimerObject = new Localizablestring($config->getCheckout()->getCheckoutDisclaimer());
        $disclaimer = !empty($disclaimerObject->get())?$disclaimerObject->get():\Joomla\CMS\Language\Text::_('COM_EVENTGALLERY_CART_CHECKOUT_ORDER_MAIL_CONFIRMATION_DISCLAIMER');


        $paymentMethod = $order->getPaymentMethod();
        $shippingMethod = $order->getShippingMethod();

        // trigger things if the payment status changed to paid
        if ($paymentMethod != null
            && $this->_selfCache->paymentstatusid != $this->paymentstatusid
            && $this->paymentstatusid==OrderStatus::TYPE_PAYMENT_PAID) {

            // trigger payment mail if allowed
            if ($paymentMethod->sendMailOnPaymentStatusChange($order))
            {
                if ($paymentMethod->doSendOrderConfirmationMailOnlyOnPaymentComplete()) {
                    $result = $this->_sendMail('new_order', $order, $config->getCheckout()->doShowVat(), $disclaimer);
                } else {
                    if ($paymentMethod->doSendMail()) {
                        $result = $this->_sendMail('paid_order', $order, $config->getCheckout()->doShowVat(), $disclaimer);
                    }
                }
            }

            // set shipping status
            /**
             * @var OrderStatusFactory $orderStatusFactory
             */
            $orderStatusFactory = OrderStatusFactory::getInstance();
            if ($shippingMethod != null && $shippingMethod->isAutomaticallyShippableIfPaid($order)) {
                $order->setShippingStatus($orderStatusFactory->getOrderStatusById(OrderStatus::TYPE_SHIPPING_SHIPPED));
            }
        }

        // trigger email if the order status changed to send.
        if ($shippingMethod != null
            && $shippingMethod->sendMailOnShippingStatusChange($order)
            && $this->_selfCache->shippingstatusid != $this->shippingstatusid
            && $this->shippingstatusid==OrderStatus::TYPE_SHIPPING_SHIPPED) {

            if ($shippingMethod->doSendMail()) {
                $result = $this->_sendMail('shipped_order', $order, $config->getCheckout()->doShowVat(), $disclaimer);
            }
        }

        return $result;
    }

    /**
     * @param String $mailtype
     * @param Order $order
     * @return mixed|string
     */
    private function _sendMail($mailtype, $order, $show_vat, $disclaimer) {
        // Load the front end language
        $language = Factory::getLanguage();
        $language->load('com_eventgallery' , JPATH_SITE.DIRECTORY_SEPARATOR.'components/com_eventgallery', $language->getTag(), true);
        $language->load('com_eventgallery' , JPATH_SITE.DIRECTORY_SEPARATOR.'language'.DIRECTORY_SEPARATOR.'overrides', $language->getTag(), true, false);
        /**
         * @var EmailTemplateManager $emailtemplateMgr
         */
        $emailtemplateMgr = EmailTemplateManager::getInstance();

        $data = Array();
        $data['disclaimer'] = $disclaimer;
        $data['order'] = $emailtemplateMgr->createOrderData($order, $show_vat);


        $data = json_decode(json_encode($data), FALSE);

        $to = Array($order->getEMail(), $order->getBillingAddress()==null? "": $order->getBillingAddress()->getFirstName().' '.$order->getBillingAddress()->getLastName());
        return $emailtemplateMgr->sendMail($mailtype, $order->getLanguage(), true, $data, $to, true);
    }

}
