<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Site\Controller;
use Joomla\CMS\Factory;
use Joomla\Filesystem\File;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Router\Route;
use Svenbluege\Component\Eventgallery\Site\Library\Common\Downloadlog;
use Svenbluege\Component\Eventgallery\Site\Library\Common\ImageProcessor;
use Svenbluege\Component\Eventgallery\Site\Library\Common\LogWorkaround;
use Svenbluege\Component\Eventgallery\Site\Library\Common\S3Client;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FileFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\OrderFactory;
use Svenbluege\Component\Eventgallery\Site\Library\File\S3File;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\FlickrFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\GooglePhotosFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\GooglePhotosSharedPageFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\LocalFolder;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\S3Folder;
use Svenbluege\Component\Eventgallery\Site\Library\ImageLineitem;
use Svenbluege\Component\Eventgallery\Site\Library\Lineitem;
use Svenbluege\Component\Eventgallery\Site\Library\Order;
use Svenbluege\Component\Eventgallery\Site\Library\OrderStatus;
use ZipArchive;

defined('_JEXEC') or die;



class DownloadController extends ResizeimageController
{

    const COM_EVENTGALLERY_LOGFILENAME = 'com_eventgallery_orderdownload.log.php';

    public function __construct($config = array())
    {
        parent::__construct($config);

        (new LogWorkaround())->registerLogger('com_eventgallery_formatted_text_logger', \Svenbluege\Component\Eventgallery\Site\Library\Common\FormattedTextLogger::class, true);
        Log::addLogger(
            array(
                'text_file' => self::COM_EVENTGALLERY_LOGFILENAME,
                'logger' => 'com_eventgallery_formatted_text_logger'
            ),
            Log::ALL,
            'com_eventgallery_orderdownload'
        );
    }

    public function loadEventgalleryConfig() {
        /**
         * @var JApplicationSite $app
         * @var \Joomla\Registry\Registry $registry
         */
        $app = Factory::getApplication();
        $params = $app->getParams();
        return \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance($params);
    }

    /**
     * @param \Svenbluege\Component\Eventgallery\Site\Library\File\File $file
     * @param string $type
     * @return void
     */
    private function trackDownload($file, $userid, $reference, $type) {
        $eventgalleryConfig = $this->loadEventgalleryConfig();

        if (!$eventgalleryConfig->getGeneral()->isDownloadLogEnabled()) {
            return;
        }

        $row = Factory::getApplication()->bootComponent('com_eventgallery')->getMVCFactory()->createTable('Downloadlog', 'Administrator');

        /**
         * @var \Joomla\Application\Web\WebClient $client
         */
        $client = \Joomla\CMS\Factory::getApplication()->client;
        $isbot = $client->robot?1:0;
        $browser = $client->userAgent;

        $row->bind([
            "folder" => $file->getFolderName(),
            "file" => $file->getFileName(),
            "type" => $type,
            "reference" => $reference,
            "userid" => $userid,
            "isbot" => $isbot,
            "browser" => $browser
        ]);

        $result = $row->store();
    }

    public function redirectToUrl($url) {
        /**
         * @var Joomla\CMS\Application\SiteApplication $app
         */
        $app = Factory::getApplication();
        $app->redirect($url);
        $app->close();
    }

    public function display($cachable = false, $urlparams = array())
    {
        /**
         * @var JApplicationSite $app
         * @var \Joomla\Registry\Registry $registry
         */
        $app = Factory::getApplication();
        $eventgalleryConfig = $this->loadEventgalleryConfig();

        $str_folder = $app->input->getString('folder', null);
        $str_file = $app->input->getString('file', null);
        $is_sharing_download = $app->input->getBool('is_for_sharing', false);

        /**
         * @var FileFactory $fileFactory
         */
        $fileFactory = FileFactory::getInstance();

        $file = $fileFactory->getFile($str_folder, $str_file);

        $allowDownloadOfOriginalImage = $file->getFolder()->doAllowDownloadOfOriginalImage($eventgalleryConfig);
        $allowDownloadAtAll = $file->getFolder()->doAllowDownloadAtAll($eventgalleryConfig);
        $redirectUrl = $eventgalleryConfig->getSocial()->getRedirectURL();

        if (!is_object($file) || !$file->isPublished()) {
            throw new \Exception(\Joomla\CMS\Language\Text::_('COM_EVENTGALLERY_SINGLEIMAGE_NO_PUBLISHED_MESSAGE'), 404);
        }

        $folder = $file->getFolder();

        // Check of the user has the permission to grab the image
        if (!$folder->isPublished() || !$folder->isVisible() || !$folder->isAccessible()) {
            throw new \Exception(\Joomla\CMS\Language\Text::_('COM_EVENTGALLERY_EVENT_NO_PUBLISHED_MESSAGE'), 404);
        }

        // if this is a sharing URL and we don't allow the download until here,
        // we allow the download but disable the shared download.
        if ($is_sharing_download && !$allowDownloadAtAll) {
            $allowDownloadOfOriginalImage = false;
            $allowDownloadAtAll = true;
        }

        if (!$allowDownloadAtAll) {
            if (empty($redirectUrl)) {
                throw new \Exception(\Joomla\CMS\Language\Text::_('COM_EVENTGALLERY_EVENT_NO_DOWNLOAD_ALLOWED'), 403);
            }
            $this->redirectToUrl($redirectUrl);
        }

        // allow the download if at least one sharing type is enabled both global and for the event
        if (
                ($eventgalleryConfig->getSocial()->doUseFacebook() && $folder->getAttribs()->get('use_social_sharing_facebook', 1)==1)
            ||  ($eventgalleryConfig->getSocial()->doUseTwitter() && $folder->getAttribs()->get('use_social_sharing_twitter', 1)==1)
            ||  ($eventgalleryConfig->getSocial()->doUseLinkedIn() && $folder->getAttribs()->get('use_social_sharing_linkedin', 1)==1)
            ||  ($eventgalleryConfig->getSocial()->doUsePinterest() && $folder->getAttribs()->get('use_social_sharing_pinterest', 1)==1)
            ||  ($eventgalleryConfig->getSocial()->doUseEmail() && $folder->getAttribs()->get('use_social_sharing_email', 1)==1)
            ||  ($eventgalleryConfig->getSocial()->doUseDownload() && $folder->getAttribs()->get('use_social_sharing_download', 1)==1)

            ) {
        	// nothing to do there since the sharing options are fine.
        } else {
            $allowDownloadOfOriginalImage = false;
        }

        $user = \Joomla\CMS\Factory::getUser();
        if ($allowDownloadOfOriginalImage) {
            $this->trackDownload($file, $user->guest ? null : $user->id, "", Downloadlog::TYPE_DOWNLOAD_ORIGINAL);
        } else {
            $this->trackDownload($file, $user->guest ? null : $user->id, "", Downloadlog::TYPE_DOWNLOAD_FULL);
        }

        if ($file->getFolder()->getFolderType()->getId() == S3Folder::ID) {
            $this->downloadS3Image($allowDownloadOfOriginalImage, $is_sharing_download, $file);
        } else if (
            $file->getFolder()->getFolderType()->getId() == GooglePhotosFolder::ID
        OR $file->getFolder()->getFolderType()->getId() == GooglePhotosSharedPageFolder::ID) {
            /**
             * @var GooglePhotosFolder $file
             */
            $this->downloadGooglePhotosOriginalImage($file);

        } else {
            $this->downloadLocalImage($allowDownloadOfOriginalImage, $is_sharing_download, $file);
        }

    }

    /**
     * @param $doProvideOriginalImage
     * @param $is_sharing_download
     * @param \Svenbluege\Component\Eventgallery\Site\Library\File\File $file
     */
    private function downloadLocalImage($doProvideOriginalImage, $is_sharing_download, $file) {

        $basename = COM_EVENTGALLERY_IMAGE_FOLDER_PATH . $file->getFolderName() . '/';

        if ( $doProvideOriginalImage ) {

            // try the path to a possible original file
            $filename = $basename. COM_EVENTGALLERY_IMAGE_ORIGINAL_SUBFOLDER .'/'.$file->getFileName();

            if (!file_exists($filename)) {
                $filename = $basename . $file->getFileName();
            }

            $mime = ($mime = getimagesize($filename)) ? $mime['mime'] : $mime;
            $size = filesize($filename);
            $fp   = fopen($filename, "rb");
            if (!($mime && $size && $fp)) {
                // Error.
                return;
            }


            header("Content-type: " . $mime);
            header("Content-Length: " . $size);
            if (!$is_sharing_download) {
                header("Content-Disposition: attachment; filename=" . $file->getFileName());
            }
            header('Content-Transfer-Encoding: binary');
            header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
            fpassthru($fp);
            fclose($fp);
            return $this->endExecution();
        } else {
            if (!$is_sharing_download) {
                header("Content-Disposition: attachment; filename=" . $file->getFileName());
            }
            header('Content-Transfer-Encoding: binary');
            header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
            $this->renderThumbnail($file->getFolderName(), $file->getFileName(), COM_EVENTGALLERY_IMAGE_ORIGINAL_MAX_WIDTH);
            return $this->endExecution();
        }
    }

    /**
     * @param $doProvideOriginalImage
     * @param $is_sharing_download
     * @param S3File $file
     */
    private function downloadS3Image($doProvideOriginalImage, $is_sharing_download, $file) {

        $s3client = S3Client::getInstance();

        $joomlaConfig   = Factory::getConfig();
        $tempFileName =  tempnam($joomlaConfig->get('tmp_path'), 'eg');

        if ( $doProvideOriginalImage ) {
            $s3client->getObjectToFile(
                $s3client->getBucketForOriginalImages(),
                $file->getFolderName() . "/" . $file->getFileName(),
                $tempFileName
            );
        } else {
            $s3client->getObjectToFile(
                $s3client->getBucketForThumbnails(),
                $file->calculateS3Key(COM_EVENTGALLERY_IMAGE_ORIGINAL_MAX_WIDTH),
                $tempFileName
            );
        }

        $mime = ($mime = getimagesize($tempFileName)) ? $mime['mime'] : $mime;
        $size = filesize($tempFileName);
        $fp   = fopen($tempFileName, "rb");
        if (!($mime && $size && $fp)) {
            // Error.
            return $this->endExecution();
        }


        header("Content-type: " . $mime);
        header("Content-Length: " . $size);
        if (!$is_sharing_download) {
            header("Content-Disposition: attachment; filename=" . $file->getFileName());
        }
        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');

        fpassthru($fp);
        fclose($fp);

        unlink($tempFileName);
        return $this->endExecution();
    }

    /**
     * @param $file GooglePhotosFolder
     */
    private function downloadGooglePhotosOriginalImage($file) {
        $joomlaConfig   = Factory::getConfig();
        $tempFileName =  tempnam($joomlaConfig->get('tmp_path'), 'eg');

        $download_file = file_get_contents($file->getGoogleImageUrl($file->getWidth()));

        $handle = fopen($tempFileName, "w");
        fwrite($handle, $download_file);
        fclose($handle);

        $mimeType = mime_content_type($tempFileName);
        $fileExtensnion = "";
        if ($mimeType == 'image/jpeg') {
            $fileExtensnion .= '.jpg';
        }

        $size = filesize($tempFileName);
        $fp   = fopen($tempFileName, "rb");
        if (!($mimeType && $size && $fp)) {
            // Error.
            return $this->endExecution();
        }

        header("Content-type: " . $mimeType);
        header("Content-Length: " . $size);

        header("Content-Disposition: attachment; filename=" . $file->getFileName().$fileExtensnion);

        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');

        fpassthru($fp);
        fclose($fp);

        unlink($tempFileName);
        return $this->endExecution();
    }

    /**
     * allow to download small thumbnails in case a user did a order
     *
     * @throws Exception
     */
    public function mailthumb() {
        $app = Factory::getApplication();

        $str_orderid = $this->input->getString('orderid', null);
        $str_lineitemid = $this->input->getString('lineitemid', null);
        $str_token = $this->input->getString('token', null);


        /**
         * @var OrderFactory $orderFactory
         */
        $orderFactory = OrderFactory::getInstance();
        $order = $orderFactory->getOrderById($str_orderid);
        if ($order == null) {
            throw new \Exception("Invalid Request.");
        }

        if ($order->getToken() != $str_token) {
            throw new \Exception("Invalid Request.");
        }

        $lineitem = $order->getLineItem($str_lineitemid);
        if ($lineitem == null) {
            throw new \Exception("Invalid Request.");
        }

        if (strcmp($str_token, $order->getToken())!=0) {
            throw new \Exception("Invalid Request.");
        }

        $file = $lineitem->getFile();

        if ($file->getFolder()->getFolderType()->getId() == LocalFolder::ID) {
            $this->renderThumbnail($file->getFolderName(), $file->getFileName(), COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_CART_WIDTH);
            return $this->endExecution();
        } else if ($file->getFolder()->getFolderType()->getId() == GooglePhotosFolder::ID
            || $file->getFolder()->getFolderType()->getId() == GooglePhotosSharedPageFolder::ID) {
            /**
             * @var GooglePhotosFolder $file
             */
            $app->redirect($file->getGoogleImageUrl(104));
        }else {
            $app->redirect($file->getThumbUrl(104, 104));
        }

    }

    /**
     * This method is used to enable the download of files.
     *
     * @throws Exception
     */
    public function order() {

        $app = Factory::getApplication();

        $str_orderid = $this->input->getString('orderid', null);
        $str_lineitemid = $this->input->getString('lineitemid', null);
        $str_token = $this->input->getString('token', null);


        /**
         * @var OrderFactory $orderFactory
         */
        $orderFactory = OrderFactory::getInstance();
        $order = $orderFactory->getOrderById($str_orderid);
        if ($order == null) {
            throw new \Exception("Invalid Request.");
        }

        if ($order->getToken() != $str_token) {
            throw new \Exception("Invalid Request.");
        }

        $lineitem = $order->getLineItem($str_lineitemid);
        if ($lineitem == null) {
            throw new \Exception("Invalid Request.");
        }

        if (strcmp($str_token, $order->getToken())!=0) {
            throw new \Exception("Invalid Request.");
        }

        if ($order->getPaymentStatus()->getId() != OrderStatus::TYPE_PAYMENT_PAID) {
            throw new \Exception("Invalid Request.");
        }

        $file = $lineitem->getFile();

        $this->trackDownload($file, $order->getUserId(), $order->getId(), Downloadlog::TYPE_PURCHASE_FILE);

        if ($file->getFolder()->getFolderType()->getId() == LocalFolder::ID) {
            $this->handleLocalOrderDownload($file, $order, $lineitem);
        } elseif ($file->getFolder()->getFolderType()->getId() == S3Folder::ID) {
            $this->handleS3OrderDownload($file, $order, $lineitem);
        } elseif ($file->getFolder()->getFolderType()->getId() == GooglePhotosFolder::ID || $file->getFolder()->getFolderType()->getId() == GooglePhotosSharedPageFolder::ID) {
            $this->handleGooglePhotosOrderDownload($file, $order, $lineitem);
        } elseif ($file->getFolder()->getFolderType()->getId() == FlickrFolder::ID) {
            $this->handleViaProxy($file->getOriginalImageUrl(true), $file, $order, $lineitem);
        } else {
            $app->redirect($file->getOriginalImageUrl());
        }

    }

    /**
     * Allows to download the order as a zip file. It simply calls the order links for each image and adds the result to the zip file
     * Why? Because the order link will handle the resizing and we don't need to do it twice.
     *
     * @throws Exception
     */
    public function zip() {
        $str_orderid = $this->input->getString('orderid', null);
        $str_token = $this->input->getString('token', null);

        /**
         * @var OrderFactory $orderFactory
         */
        $orderFactory = OrderFactory::getInstance();
        $order = $orderFactory->getOrderById($str_orderid);
        if ($order == null) {
            throw new \Exception("Invalid Request.");
        }

        if ($order->getToken() != $str_token) {
            throw new \Exception("Invalid Request.");
        }

        if ($order->getPaymentStatus()->getId() != OrderStatus::TYPE_PAYMENT_PAID) {
            throw new \Exception("Invalid Request.");
        }

        $zip = new ZipArchive();
        $joomlaConfig   = Factory::getConfig();

        $tmpZipFilename = tempnam($joomlaConfig->get('tmp_path'), 'eg');


        if ($zip->open($tmpZipFilename, ZipArchive::OVERWRITE)!==TRUE) {
            exit("cannot open <$tmpZipFilename>\n");
        }

        foreach($order->getLineItems() as $lineitem) {


            /**
             * @var ImageLineitem $lineitem
             */

            if ($lineitem->getImageType()->isDigital()) {
                $file = $lineitem->getFile();

                $context = stream_context_create(
                    array(
                        'http' => array(
                            'follow_location' => true
                        )
                    )
                );

                $url = str_replace("/administrator", "",  Route::_("index.php?option=com_eventgallery&view=download&task=order&orderid=" . $order->getId() . "&lineitemid=" . $lineitem->getId() . "&token=" . $order->getToken(), false, -1));
                $download_file = false;
                $http_response_header = null;

                try{
                    $http = HttpFactory::getHttp();
                    $http->setOption('follow_location', true);
                    $response = $http->get($url);
                    $download_file = $response->body;
                    $http_response_header = $response->headers;

                } catch (\Exception $e) {
                    Log::add('error while getting image from url ' . $url . ' Error message: ' . $e->getMessage(), Log::INFO, 'com_eventgallery_orderdownload');
                }

                if ($download_file !== false) {

                    $fileuri = $file->getFolderName() . '/' . $order->getDocumentNumber(). '-' . $lineitem->getImageType()->getId() . '-' . $file->getFileName();

                    if (in_array($file->getFolder()->getFolderType()->getId(), [FlickrFolder::ID, GooglePhotosFolder::ID, GooglePhotosSharedPageFolder::ID])) {
                        $tmpfname = tempnam($joomlaConfig->get('tmp_path'), 'eg_dl');
                        $handle = fopen($tmpfname, "w");
                        fwrite($handle, $download_file);
                        fclose($handle);

                        $mimeType = mime_content_type($tmpfname);
                        if ($mimeType == 'image/jpeg') {
                            $fileuri .= '.jpg';
                        }

                        unlink($tmpfname);
                    }

                    $zip->addFromString($fileuri, $download_file);
                }
                unset($download_file);

            }

        }

        $zip->close();


        $size = filesize($tmpZipFilename);
        $fp   = fopen($tmpZipFilename, "rb");
        if (!($size && $fp)) {
            echo "Can't read zip file.";
            return $this->endExecution();
        }

        header("Pragma: public");
        header("Expires: 0");
        header("Cache-Control: must-revalidate, post-check=0, pre-check=0");
        header("Cache-Control: public");
        header("Content-Description: File Transfer");
        header("Content-type: application/octet-stream");
        header("Content-Disposition: attachment; filename=" . $order->getDocumentNumber(). '.zip');
        header("Content-Transfer-Encoding: binary");
        header("Content-Length: ".$size);

        fpassthru($fp);
        fclose($fp);
        unlink($tmpZipFilename);
        return $this->endExecution();
    }


    /**
     * @param \Svenbluege\Component\Eventgallery\Site\Library\File\File $file
     * @param Order $order
     * @param ImageLineitem $lineitem
     */
    private function handleS3OrderDownload($file, $order, $lineitem) {
        $s3client = S3Client::getInstance();
        $tempFileName = JPATH_CACHE . '/' . $file->getFileName();
        $image_thumb_file = null;
        $s3client->getObjectToFile(
            $s3client->getBucketForOriginalImages(),
            $file->getFolderName() . "/" . $file->getFileName(),
            $tempFileName);

        $imageSize = intval($lineitem->getImageType()->getSize());
        $watermark = $lineitem->getImageType()->getWatermark();

        if ($imageSize>0 || $watermark!=null) {
            $imageProcessor = new ImageProcessor();
            $imageProcessor->loadImage($tempFileName);

            $joomlaConfig   = Factory::getConfig();
            $image_thumb_file = tempnam($joomlaConfig->get('tmp_path'), 'eg') . '.' . pathinfo($file->getFileName(), PATHINFO_EXTENSION);

            $imageProcessor->setTargetImageSize($imageSize>0?$imageSize:$file->getWidth(), -1, false);
            $imageProcessor->processImage(
                false,
                $watermark!=null,
                $watermark,
                false,
                false,
                null
            );
            $imageProcessor->saveThumbnail($image_thumb_file, 90);
            $imageProcessor->copyICCProfile($tempFileName, $image_thumb_file);
            $filename = $image_thumb_file;
        } else {
            $filename = $tempFileName;
        }

        $downloadFileName = File::makeSafe($order->getDocumentNumber(). '-' . $lineitem->getId() . '-' . $lineitem->getImageType()->getDisplayName(). '-' . $file->getFileName());

        header("Content-Disposition: attachment; filename=" . $downloadFileName);
        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');

        $mime = ($mime = getimagesize($filename)) ? $mime['mime'] : $mime;
        $size = filesize($filename);
        $fp   = fopen($filename, "rb");
        if (!($mime && $size && $fp)) {
            // Error.
            return $this->endExecution();
        }

        header("Content-type: " . $mime);
        header("Content-Length: " . $size);
        fpassthru($fp);
        fclose($fp);

        if ($image_thumb_file != null) {
            unlink($image_thumb_file);
        }

        unlink($tempFileName);
        return $this->endExecution();
    }

    /**
     * @param \Svenbluege\Component\Eventgallery\Site\Library\File\File $file
     * @param Order $order
     * @param ImageLineitem $lineitem
     * @throws Exception
     */
    private function handleLocalOrderDownload($file, $order, $lineitem) {

        $imageSize = intval($lineitem->getImageType()->getSize());

        $downloadFileName = File::makeSafe($order->getDocumentNumber(). '-' . $lineitem->getId() . '-' . $lineitem->getImageType()->getDisplayName(). '-' . $file->getFileName());

        header("Content-Disposition: attachment; filename=" . $downloadFileName);
        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');

        $watermark = $lineitem->getImageType()->getWatermark();
        if ($imageSize>0 || $watermark != null) {
            $this->renderThumbnail($file->getFolderName(), $file->getFileName(), $imageSize>0?$imageSize:$file->getWidth() , false, false, $watermark!=null, $watermark, false);
            return $this->endExecution();
        }

        $fileContent = $file->getOriginalFile();

        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->buffer($fileContent);

        header("Content-type: " . $mimeType);
        header("Content-Length: " .strlen($fileContent));
        echo $fileContent;

        return $this->endExecution();
    }

    /**
     * @param GooglePhotosFolder $file
     * @param Order $order
     * @param ImageLineitem $lineitem
     * @throws Exception
     */
    private function handleGooglePhotosOrderDownload($file, $order, $lineitem) {

        $url = $file->getGoogleImageUrl($file->getWidth());
        $this->handleViaProxy($url, $file, $order, $lineitem);
    }

    /**
     * @param $file \Svenbluege\Component\Eventgallery\Site\Library\File\File
     * @param $order Order
     * @param $lineitem Lineitem
     * @return void
     */
    public function handleViaProxy($url, $file, $order, $lineitem) {

        try{
            $http = HttpFactory::getHttp();
            $http->setOption('follow_location', false);
            $response = $http->get($url);
            $download_file = $response->body;
            $http_response_header = $response->headers;

        } catch (\Exception $e) {
            Log::add('handleViaProxy: error while getting image from url ' . $url . ' Error message: ' . $e->getMessage(), Log::INFO, 'com_eventgallery_orderdownload');
        }

        $joomlaConfig   = Factory::getConfig();

        $fileExtension = '';
        if ($http_response_header['content-type'] == 'image/jpeg') {
            $fileExtension.='.jpg';
        } else {
            $tempFileNameForMimeType = tempnam($joomlaConfig->get('tmp_path'), 'eg_dl');
            $handle = fopen($tempFileNameForMimeType, "w");
            fwrite($handle, $download_file);
            fclose($handle);

            $mimeType = mime_content_type($tempFileNameForMimeType);
            if ($mimeType == 'image/jpeg') {
                $fileExtension = '.jpg';
            }

            unlink($tempFileNameForMimeType);
        }
        $tempFileName = tempnam($joomlaConfig->get('tmp_path'), 'eg_dl').$fileExtension;
        $handle = fopen($tempFileName, "w");
        fwrite($handle, $download_file);
        fclose($handle);

        $imageSize = intval($lineitem->getImageType()->getSize());
        $watermark = $lineitem->getImageType()->getWatermark();

        if ($imageSize>0 || $watermark!=null) {
            $imageProcessor = new ImageProcessor();
            $imageProcessor->loadImage($tempFileName);

            $image_thumb_file = tempnam($joomlaConfig->get('tmp_path'), 'eg') . $fileExtension;

            $imageProcessor->setTargetImageSize($imageSize>0?$imageSize:$file->getWidth(), -1, false);
            $imageProcessor->processImage(
                false,
                $watermark!=null,
                $watermark,
                false,
                false,
                null
            );
            $imageProcessor->saveThumbnail($image_thumb_file, 90);
            $imageProcessor->copyICCProfile($tempFileName, $image_thumb_file);
            $filename = $image_thumb_file;
        } else {
            $filename = $tempFileName;
        }

        $downloadFileName = File::makeSafe($order->getDocumentNumber(). '-' . $lineitem->getId() . '-' . $lineitem->getImageType()->getDisplayName(). '-' . $file->getFileName() . $fileExtension);
        header("Content-Disposition: attachment; filename=" . $downloadFileName);
        header('Content-Transfer-Encoding: binary');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');

        $mime = ($mime = getimagesize($filename)) ? $mime['mime'] : $mime;
        $size = filesize($filename);
        $fp   = fopen($filename, "rb");
        if (!($mime && $size && $fp)) {
            // Error.
            return $this->endExecution();
        }

        header("Content-type: " . $mime);
        header("Content-Length: " . $size);
        fpassthru($fp);
        fclose($fp);

        if ($image_thumb_file != null) {
            unlink($image_thumb_file);
        }

        unlink($tempFileName);
        return $this->endExecution();
    }

}
