<?php

namespace Svenbluege\Component\Eventgallery\Site\Library\Common;

use Svenbluege\Component\Eventgallery\Site\Library\Common\Css\Breakpoint;

class CSSRenderer
{

    public static function renderStyleTag(?string $json, string $context): string {
        $renderer = new CSSRenderer();

        if (empty($json)) {
            return '';
        }

        try {
            $breakpoints = $renderer->parseJSON($json);
            $css = $renderer->renderBreakpoints($breakpoints, $context);
            return "<style>$css</style>";
        } catch (\InvalidArgumentException $e) {

        }
        return '';
    }
    /**
     * @param Breakpoint[] $breakpoints
     * @param string $context
     * @return string
     */
    public function renderBreakpoints(array $breakpoints, string $context): string {
        $css = [];

        foreach ($breakpoints as $breakpoint) {
            $css[] = $this->renderBreakpoint($breakpoint, $context);
        }

        return implode(PHP_EOL . PHP_EOL, $css);

    }

    /**
     * Generiert CSS für einen einzelnen Breakpoint
     *
     * @param Breakpoint $breakpoint Der zu verarbeitende Breakpoint
     * @param string $context Der CSS-Selektor
     * @return string Das generierte CSS mit Media Query
     */
    public function renderBreakpoint(Breakpoint $breakpoint, string $context): string
    {
        $width = $this->truncate(100 / $breakpoint->itemsPerRow);

        return sprintf(
            '@media (min-width: %dpx) {%s {width: %s%%;}}',
            $breakpoint->width,
            $context,
            $width
        );
    }

    /**
     * @param float $number
     * @param int $decimals
     * @return float
     */
    private function truncate(float $number, int $decimals = 2): float
    {
        $tempResult = floor($number * 10**$decimals);
        return $tempResult / 10**$decimals;
    }


    /**
     * @param string $json
     * @return Breakpoint[]
     * @throws JsonException
     * @throws \InvalidArgumentException
     */
    public function parseJSON(string $json): array
    {
        $data = json_decode($json, true, 512, JSON_THROW_ON_ERROR);

        if (!isset($data['breakpoints'])) {
            throw new \InvalidArgumentException('JSON is valid, but there are no breakpoints');
        }

        return array_map(function(array $item) {
            if (!array_key_exists('width', $item) || !array_key_exists('itemsPerRow', $item)) {
                throw new \InvalidArgumentException('Missing required fields for width or itemsPerRow');
            }

            if ($item['width'] === null || $item['itemsPerRow'] === null) {
                throw new \InvalidArgumentException('Missing required fields values for width or itemsPerRow');
            }

            $breakpoint = new Breakpoint();
            $breakpoint->width = (int)$item['width'];
            $breakpoint->itemsPerRow = (int)$item['itemsPerRow'];
            return $breakpoint;
        }, $data['breakpoints']);
    }

}