<?php

namespace Svenbluege\Component\Eventgallery\Site\Library\Connector;

use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\Exception\ExecutionFailureException;
use Joomla\Http\Response;
use Joomla\Session\SessionInterface;
use Svenbluege\Component\Eventgallery\Site\Library\Common\LogWorkaround;
use Svenbluege\Component\Eventgallery\Site\Library\Data\Exif;
use Svenbluege\Component\Eventgallery\Site\Library\File\File;
use Svenbluege\Component\Eventgallery\Site\Library\GooglePhotosAccount;

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

class GooglePhotosConnector
{

    /**
     * This cache is necessary to avoid requesting an auth token too many times. This happens
     * if the refresh token is invalid but not empty.
     *
     * @var array
     */
    static $tokenCache = [];

    public static $cachebasedir;
    public static $cache_life = '86400'; //caching time, in seconds
    public static $requesttimeout = 5;


    /**
     * @return string
     */
    const COM_EVENTGALLERY_GOOGLEPHOTOS_LOGFILENAME = 'com_eventgallery_googlephotos.log.php';

    public static function doRequest($cachelifetime, $method, $url, $hash, $data, $doCache)
    {
        self::initializeLogger();

        self::initCacheDirs();

        $cachefilename = self::$cachebasedir . $hash . '.json';

        if ($doCache && file_exists($cachefilename) && (time() - filemtime($cachefilename) <= $cachelifetime)) {
            // no need to do anything since the cache is still valid

        } else {

            //Log::add("Doing a request for $url", Log::INFO, 'com_eventgallery');
            /**
             * @var Response $result
             */

            try {
                if ($method == 'POST') {
                    $result = HttpFactory::getHttp()->post($url, $data, ['Content-Type' => 'application/json'], self::$requesttimeout);
                } else {
                    $result = HttpFactory::getHttp()->get($url, ['Content-Type' => 'application/json'], self::$requesttimeout);
                }

                $body = $result->body;

                if ($result->code < 300) {
                    $fh = fopen($cachefilename, 'w') or die("can't open file $cachefilename");
                    fwrite($fh, $body);
                    fclose($fh);
                } else {
                    if ($result->code == 400) {
                        Log::add("Request answered with HTTP 400 for URL: $url\n". $body, Log::ERROR, 'com_eventgallery');
                    } else {
                        Log::add('Unable to connect to remote host. Make sure curl is available or allow_url_fopen=On and the server has access to the internet. url: ' . $url, Log::ERROR, 'com_eventgallery');
                    }
                }

                // make sure we don't cause reload issues by loading elements too often. Max every minute should do.
                if (empty($body)) {
                    Log::add("Invalid return result detected. The response body is empty for url $url. Please check the used credentials.", Log::ERROR, 'com_eventgallery');
                    touch($cachefilename, time() - (self::$cache_life - 60)) ;
                } else if (self::verifyResult($body) == false) {
                    Log::add("Invalid return result detected. The response body is not a valid JSON result for url $url. Please check the used credentials. $body", Log::ERROR, 'com_eventgallery');
                    touch($cachefilename, time() - (self::$cache_life - 60)) ;
                }
                $result = NULL;

            } catch (\RuntimeException $e) {
                Log::add('Error for url '. $url .': '.$e->getMessage(), Log::ERROR, 'com_eventgallery');
            }

        }

        return $cachefilename;

    }

    /**
     * @param $responseBody
     * @return bool
     */
    public static function verifyResult($responseBody) {
        $json = json_decode($responseBody);
        return json_last_error() === JSON_ERROR_NONE;
    }

    public static function initCacheDirs() {

        if (!is_dir(JPATH_CACHE)) {
            mkdir(JPATH_CACHE);
        }

        self::$cachebasedir = COM_EVENTGALLERY_GOOGLE_PHOTOS_CACHE_PATH;

        if (!is_dir(self::$cachebasedir)) {
            mkdir(self::$cachebasedir);
        }
    }

    /**
     * Updates the album with the database
     * @param $cachelifetime
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @param $db DatabaseDriver
     * @return null
     */
    public static function syncAlbums($cachelifetime, $api_clientid, $api_secret, $refresh_token, $db)
    {
        self::initCacheDirs();

        $hash_prefix = md5( $api_clientid.$api_secret.$refresh_token);
        $filename =  $hash_prefix . '.albums.obj';
        $serOBjectPath = self::$cachebasedir . $filename;


        if (file_exists($serOBjectPath) && (time() - filemtime($serOBjectPath) <= $cachelifetime)) {
            return null;
        }

        $access_token = self::getAccessToken($db, $api_clientid, $api_secret, $refresh_token);
        if ($access_token == "") {
            return null;
        }

        Log::add("Syncing albums for the clientID $api_clientid.", Log::INFO, 'com_eventgallery');

        $url = 'https://photoslibrary.googleapis.com/v1/albums/?pageSize=50&prettyPrint=true';
        $url .= "&access_token=" . $access_token;

        $currentPageNumber = 1;
        $cachefilename = self::doRequest($cachelifetime, 'GET', $url, $hash_prefix.'.albums.'.$currentPageNumber , null, true);
        $jsonAlbums = json_decode(@file_get_contents($cachefilename));

        $json_error = json_last_error();
        if ($json_error != JSON_ERROR_NONE) {
            Log::add("Unable to load json content from file. File Name: $cachefilename for albums. Error: $json_error", Log::ERROR, 'com_eventgallery');
            return;
        }

        $albums = $jsonAlbums->albums;

        if (!is_array($albums)) {
            Log::add("The content of the file is not an array. File Name: $cachefilename for albums. Error: $json_error", Log::ERROR, 'com_eventgallery');
            return;
        }

        while (!empty($jsonAlbums->nextPageToken)) {
            $currentPageNumber++;
            $pagingUrl = $url . '&pageToken=' . $jsonAlbums->nextPageToken;
            $cachefilename = self::doRequest($cachelifetime, 'GET', $pagingUrl, $hash_prefix.'.albums.'.$currentPageNumber, null, true);
            $jsonAlbums = json_decode(file_get_contents($cachefilename));
            $json_error = json_last_error();
            if ($json_error != JSON_ERROR_NONE) {
                Log::add("Unable to load json content from file. File Name: $cachefilename for albums. Error: $json_error", Log::ERROR, 'com_eventgallery');
                return;
            }

            if (isset($jsonAlbums->albums)) {
                $albums = array_merge($albums, $jsonAlbums->albums);
            }
        }

        foreach($albums as $album) {
            $query = $db->getQuery(true);
            $query->update('#__eventgallery_file')
                ->set('googlephotos_baseurl='.$db->quote("".$album->coverPhotoBaseUrl))
                ->where('folder='.$db->quote($album->id) .' AND ismainimage=1');
            $db->setQuery($query)->execute();

        }

        $c = 'just dummy content';//serialize($album);
        file_put_contents($serOBjectPath, $c);

        return;
    }

    /**
     * retrieved albums from Google
     * @param $cachelifetime
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @param $db DatabaseDriver
     * @return null
     */
    public static function getAlbums($cachelifetime, $api_clientid, $api_secret, $refresh_token, $db) {
        self::initCacheDirs();

        $hash_prefix = md5( $api_clientid.$api_secret.$refresh_token);

        $access_token = self::getAccessToken($db, $api_clientid, $api_secret, $refresh_token);

        $url = 'https://photoslibrary.googleapis.com/v1/albums/?pageSize=50&prettyPrint=true';
        $url .= "&access_token=" . $access_token;

        $currentPageNumber = 1;
        $cachefilename = self::doRequest($cachelifetime, 'GET', $url, $hash_prefix.'.albums.'.$currentPageNumber , null, true);
        $jsonAlbums = json_decode(@file_get_contents($cachefilename));

        $json_error = json_last_error();
        if ($json_error != JSON_ERROR_NONE) {
            Log::add("Unable to load json content from file. File Name: $cachefilename for albums. Error: $json_error", Log::ERROR, 'com_eventgallery');
            return;
        }

        $albums = $jsonAlbums->albums;
        if (!is_array($albums)) {
            $albums = [];
        }

        while (!empty($jsonAlbums->nextPageToken)) {
            $currentPageNumber++;
            $pagingUrl = $url . '&pageToken=' . $jsonAlbums->nextPageToken;
            $cachefilename = self::doRequest($cachelifetime, 'GET', $pagingUrl, $hash_prefix.'.albums.'.$currentPageNumber, null, true);
            $jsonAlbums = json_decode(file_get_contents($cachefilename));
            if (isset($jsonAlbums->albums)) {
                $albums = array_merge($albums, $jsonAlbums->albums);
            }
        }

        return $albums;

    }

    /**
     * Updates the album with the database
     * @param $cachelifetime
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @param $db DatabaseDriver
     * @param $albumId
     * @return null
     */
    public static function syncAlbum($cachelifetime, $api_clientid, $api_secret, $refresh_token, $db, $albumId)
    {
        $startTime = microtime(true);
        self::initCacheDirs();

        $serOBjectPath = self::$cachebasedir . $albumId .'.obj';

        if (file_exists($serOBjectPath) && (time() - filemtime($serOBjectPath) <= $cachelifetime)) {
            return null;
        }

        $access_token = self::getAccessToken($db, $api_clientid, $api_secret, $refresh_token);

        if ($access_token == "") {
            return null;
        }

        Log::add("Syncing the album $albumId for the clientID $api_clientid.", Log::INFO, 'com_eventgallery');

        $hash_prefix = $albumId;

        $url = 'https://photoslibrary.googleapis.com/v1/albums/'. $albumId .'?prettyPrint=true';
        $url .= "&access_token=" . $access_token;

        $cachefilename = self::doRequest($cachelifetime, 'GET', $url, $hash_prefix.'.album' , null, true);
        $jsonAlbum = json_decode(@file_get_contents($cachefilename));

        $currentPageNumber = 1;
        $url = 'https://photoslibrary.googleapis.com/v1/mediaItems:search?pageSize=100&prettyPrint=true';
        $url .= "&access_token=" . $access_token;
        $data = json_encode(
            [
                'albumId' => $albumId
            ]);
        $cachefilename = self::doRequest($cachelifetime, 'POST', $url, $hash_prefix.'.items.'.$currentPageNumber, $data, true);

        $jsonMediaItems = json_decode(@file_get_contents($cachefilename));

        $json_error = json_last_error();
        if ($json_error != JSON_ERROR_NONE) {
            Log::add("Unable to load json content from file. File Name: $cachefilename for $albumId. Error: $json_error", Log::ERROR, 'com_eventgallery');
            $album['photos'] = [];
            $album['overallCount'] = 0;
            return (object)$album;
        }

        $mediaItems = $jsonMediaItems->mediaItems;

        while (!empty($jsonMediaItems->nextPageToken)) {
            $currentPageNumber++;
            $pagingUrl = $url . '&pageToken=' . $jsonMediaItems->nextPageToken;
            $cachefilename = self::doRequest($cachelifetime, 'POST', $pagingUrl, $hash_prefix.'.items.'.$currentPageNumber, $data, true);
            $jsonMediaItems = json_decode(file_get_contents($cachefilename));
            if (is_array($jsonMediaItems->mediaItems)) {
                $mediaItems = array_merge($mediaItems, $jsonMediaItems->mediaItems);
            }
        }


        $album = Array();
        $photos = Array();
        $ordering = 0;
        #echo "After DOM loaded:". memory_get_usage() . "<br>\n";

        foreach ($mediaItems as $mediaItem) {

            $photo = Array();
            if (isset($mediaItem->mediaMetadata->photo)) {
                $photo['type'] = File::TYPE_IMAGE;
            } else {
                $photo['type'] = File::TYPE_VIDEO;
            }

            $photo['baseurl'] = $mediaItem->baseUrl;
            $photo['width'] = $mediaItem->mediaMetadata->width;
            $photo['height'] = $mediaItem->mediaMetadata->height;

            $photo['caption'] = isset($mediaItem->description)?$mediaItem->description:'';
            $photo['title'] = "";//$mediaItem->filename;
            $photo['date'] = $mediaItem->mediaMetadata->creationTime;
            $photo['folder'] = $albumId;
            $photo['file'] = $mediaItem->id;
            $photo['googlephotos_filename'] = $mediaItem->filename;


            $exif = Array();

            $exif['fstop'] = isset($mediaItem->mediaMetadata->photo->apertureFNumber)? $mediaItem->mediaMetadata->photo->apertureFNumber:'';
            $exif['focallength'] = isset($mediaItem->mediaMetadata->photo->focalLength)?$mediaItem->mediaMetadata->photo->focalLength:'';
            $exif['exposuretime'] = isset($mediaItem->mediaMetadata->photo->exposureTime)?Exif::float2rat((float)$mediaItem->mediaMetadata->photo->exposureTime):'';
            $exif['model'] = (isset($mediaItem->mediaMetadata->photo->cameraModel)?$mediaItem->mediaMetadata->photo->cameraModel:'');
            $exif['iso'] = isset($mediaItem->mediaMetadata->photo->isoEquivalent)?$mediaItem->mediaMetadata->photo->isoEquivalent:'';
            $exif['filename'] = isset($mediaItem->filename)?$mediaItem->filename:'';

            $photo['exif'] = (object)$exif;
            $photo['published'] = 1;

            $photo['ordering'] = $ordering++;

            $photos[$photo['file']] = $photo;
            unset($photo);
        }

        $album['folder'] = $albumId;
        $album['file'] = $jsonAlbum->coverPhotoMediaItemId;
        $album['photos'] = $photos;
        $album['overallCount'] = $jsonAlbum->mediaItemsCount;

        $album['description'] = $jsonAlbum->title;

        $album['width'] = 1440;
        $album['height'] = 1440;

        $album['title'] = $jsonAlbum->title;
        $album['baseurl'] = $jsonAlbum->coverPhotoBaseUrl;

        #echo "Finally:". memory_get_usage() . "\n<br>";
        #echo memory_get_peak_usage() . "\n<br>";

        $start = microtime(true);

        $album = (object)$album;

        $db->transactionStart();

        try {

            $query = $db->getQuery(true);
            $query->select('id, file')->from('#__eventgallery_file')
                ->where('folder='.$db->quote($albumId));
            $db->setQuery($query);
            $oldFiles = $db->loadObjectList();

            $oldFileIDs = [];
            if (is_iterable($oldFiles)) {
                foreach ($oldFiles as $oldFile) {
                    $oldFileIDs[$oldFile->file] = $oldFile->id;
                }
            }

            $query = $db->getQuery(true);
            $query->delete('#__eventgallery_file')
                ->where('folder='.$db->quote($albumId));
            $db->setQuery($query);
            $db->execute();

            if (count($photos)>0) {

                foreach(array_chunk($photos, 100) as $photosChunk) {
                    self::addPhotosToDatabase($photosChunk, $albumId, $db, $oldFileIDs);
                }

                $query = $db->getQuery(true);
                $query->update('#__eventgallery_file')
                    ->set('ismainimage=1')
                    ->where('folder=' . $db->quote($album->folder) . ' AND file=' . $db->quote($album->file));
                $db->setQuery($query);
                $db->execute();

                $query = $db->getQuery(true);
                $query->update('#__eventgallery_folder')
                    ->set('googlephotostitle=' . $db->quote($album->title))
                    ->where('folder=' . $db->quote($album->folder));
                $db->setQuery($query);
                $db->execute();

            }
            $db->transactionCommit();

        } catch (ExecutionFailureException $e) {
            Log::add('Catched database execution while updating Google Photos files. Error message: '. $e->getMessage(), Log::ERROR, 'com_eventgallery');
            $db->transactionRollback();
        }

        $c = 'just dummy content because we want to remember the last update time.';
        file_put_contents($serOBjectPath, $c);

        $endTime = microtime(true);

        // no need to return anything since this is just an update message.
        return null;
    }

    private static function addPhotosToDatabase($photos, $albumId, $db, $oldFileIDs) {

        $query = $db->getQuery(true);

        $query->insert($db->quoteName('#__eventgallery_file'))
            ->columns(
                       'id, 
                        folder,
                        file,
                        googlephotos_filename,
                        width,
                        height,
                        caption,
                        title,
                        googlephotos_baseurl,
                        url,
                        exif,
                        ordering,
                        ismainimage,
                        ismainimageonly,
                        type,
                        hits,
                        published,
                        userid,
                        modified,
                        created,
                        creation_date'
            );


        foreach ($photos as $photo) {
            $query->values(implode(',', array(
                isset($oldFileIDs[$photo['file']]) ? $db->quote((int)$oldFileIDs[$photo['file']]):0,
                $db->quote($albumId),
                $db->quote($photo['file']),
                $db->quote($photo['googlephotos_filename']),
                $db->quote($photo['width']),
                $db->quote($photo['height']),
                $db->quote($photo['title']),
                $db->quote($photo['caption']),
                $db->quote($photo['baseurl']),
                $db->quote(''),
                $db->quote(json_encode($photo['exif'])),
                $db->quote($photo['ordering']),
                0,
                0,
                $db->quote($photo['type']),
                0,
                1,
                0,
                'now()',
                'now()',
                $db->quote(date('YmdHis', strtotime($photo['date'])))
            )));

        }

        $db->setQuery($query);
        $db->execute();
    }

    private static function createPickerSession($token) {
        $url = "https://photospicker.googleapis.com/v1/sessions";
        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $token
        ];

        $response = \Joomla\CMS\Http\HttpFactory::getHttp()->post($url, [], $headers , 3);

        if ($response->code == 200) {
            $response_body = json_decode($response->body);
            return $response_body;
       } else {
            $response_body = json_decode($response->body);
            return false;
        }
    }

    public static function getPickerSession($db, GooglePhotosAccount $account, bool $doNewSession) {
        $token = self::getAccessToken($db, $account->getClientId(), $account->getSecret(), $account->getRefreshToken());
        $session = Factory::getApplication()->getSession();
        $pickerSession = null;

        if (!$doNewSession) {
            $pickerSession = $session->get('google-picker-session', null);
        }

        if ($pickerSession == null) {
            $pickerSession = self::createPickerSession($token);
            $session->set('google-picker-session', $pickerSession);
            if ($pickerSession == false) {
                return null;
            }
        }

        $url = "https://photospicker.googleapis.com/v1/sessions/".$pickerSession->id;
        $headers = [
            'Content-Type' =>  'application/json',
            'Authorization' => 'Bearer ' . $token
        ];
        $response = \Joomla\CMS\Http\HttpFactory::getHttp()->get($url, $headers , 3);

        if ($response->code == 200) {
            $pickerSession = json_decode($response->body);
            $session->set('google-picker-session', $pickerSession);
        } else {
            $response_body = json_decode($response->body);
            return false;
        }

        return $pickerSession;
    }

    private static function fetchImagesPage($token, $itemsQuery, $pageToken) {

        $currentItemsQuery = $itemsQuery;

        if(!empty($pageToken)) {
            $currentItemsQuery .= "&pageToken=$pageToken";
        }

        $url = "https://photospicker.googleapis.com/v1/mediaItems?" . $currentItemsQuery;
        $headers = [
            'Content-Type' =>  'application/json',
            'Authorization' => 'Bearer ' . $token
        ];
        $response = \Joomla\CMS\Http\HttpFactory::getHttp()->get($url, $headers , 3);

        $resonse_body = json_decode($response->body);

        return $resonse_body;
    }
    public static function fetchImages($db, GooglePhotosAccount $account, $pageToken) {
        $token = self::getAccessToken($db, $account->getClientId(), $account->getSecret(), $account->getRefreshToken());
        $session = Factory::getApplication()->getSession();
        $pickerSession = $session->get('google-picker-session', null);

        $pageSize = 100; // user definable; default up to 100

        $itemsQuery ="sessionId={$pickerSession->id}&pageSize=$pageSize";

        $response = null;
        $mediaItems = [];

        do {
            $response = self::fetchImagesPage($token, $itemsQuery, $pageToken);
            $mediaItems = array_merge($mediaItems, $response->mediaItems);
            $pageToken = isset($response->nextPageToken) ? $response->nextPageToken : '';
        } while (!empty($pageToken) || EVENTGALLERY_EXTENDED != true);

        $result = new \stdClass();
        $result->mediaItems = $mediaItems;
        $result->pageSize = $pageSize;


        if (EVENTGALLERY_EXTENDED) {
            return $result;
        } else {
            return array_slice($result, 0, 30);
        }

    }

    public static function fetchImage($db, GooglePhotosAccount $account, $url) {
        $token = self::getAccessToken($db, $account->getClientId(), $account->getSecret(), $account->getRefreshToken());
        $session = Factory::getApplication()->getSession();
        $pickerSession = $session->get('google-picker-session', null);

        $headers = [
            'Authorization' => 'Bearer ' . $token
        ];
        $response = \Joomla\CMS\Http\HttpFactory::getHttp()->get($url, $headers , 3);

        $result = new \stdClass();
        $result->imageData = $response->getBody();
        $result->contentType = $response->getHeader('Content-Type')[0];
        $result->contentDisposition = $response->getHeader('Content-Disposition')[0];
        return $result;

    }

    /**
     * use a refresh token to create an access token.
     *
     * @param $db DatabaseDriver
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @return mixed|string
     */
    private static function getAccessToken($db, $api_clientid, $api_secret, $refresh_token)
    {
        self::initializeLogger();

        $refreshTokenHash = hash('sha256', $refresh_token);

        if (isset(self::$tokenCache[$refreshTokenHash])) {
            return self::$tokenCache[$refreshTokenHash];
        }

        $query = $db->getQuery(true);
        $query->select('access_token')
            ->from('#__eventgallery_auth_token')
            ->where('valid_until > now()')
            ->andWhere('refresh_token_hash='.$db->quote($refreshTokenHash));
        $db->setQuery($query);
        $access_token = $db->loadResult();

        if (!empty($access_token)) {
            self::$tokenCache[$refreshTokenHash] = $access_token;
            return $access_token;
        }

        try {
            $response = self::requestAccessToken($api_clientid, $api_secret, $refresh_token);
        } catch (\Exception $e) {
            Log::add('Unable to execute requestAccessToken to get access token from Google. Check the refresh token for this account. Error message: ' . $e->getMessage(), Log::ERROR, 'com_eventgallery');
        }

        /**
         * Save the stuff to the database and set the timestamp to the future. Remove some seconds so we avoid edge cases.
         */
        $access_token = "";

        if (isset($response->access_token)) {
            $db->transactionStart();
            try {
                $access_token = $response->access_token;
                $validityTime = (int)$response->expires_in - 120;

                $query = $db->getQuery(true);
                $query->delete()->from('#__eventgallery_auth_token')
                    ->where('refresh_token_hash=' . $db->quote($refreshTokenHash));
                $db->setQuery($query);
                $db->execute();

                $query = $db->getQuery(true);
                $query->insert('#__eventgallery_auth_token')
                    ->columns(array('refresh_token_hash', 'access_token', 'valid_until'))
                    ->values(implode(',', array(
                        $db->quote($refreshTokenHash), $db->quote($access_token), 'DATE_ADD(NOW(), INTERVAL ' . (int)$validityTime . ' SECOND)')));
                $db->setQuery($query);
                $db->execute();

                $db->transactionCommit();
            } catch (ExecutionFailureException $e) {
                Log::add('Catched database execution while adding token to database. Error message: '. $e->getMessage(), Log::ERROR, 'com_eventgallery');
                $db->transactionRollback();
            }
        } else {
            Log::add("Unable to get access token from Google. Check the refresh token for this account with the clientID $api_clientid. Error message: ". (isset($response)?$response->error:''). ', ' . (isset($response)?$response->error_description:''), Log::ERROR, 'com_eventgallery');
        }

        self::$tokenCache[$refreshTokenHash] = $access_token;

        return $access_token;
    }

    /**
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @return array
     */
    public static function requestAccessToken($api_clientid, $api_secret, $refresh_token)
    {
        $data = [
            'client_id' => $api_clientid,
            'client_secret' => $api_secret,
            'refresh_token' => $refresh_token,
            'grant_type' => 'refresh_token'
        ];

        $response = null;

        $response = HttpFactory::getHttp()->post('https://www.googleapis.com/oauth2/v4/token', $data, [], 2);
        $body = $response->body;
        $response = json_decode($body);

        return $response;
    }

    /**
     * @return void
     */
    private static function initializeLogger(): void
    {
        (new LogWorkaround())->registerLogger('com_eventgallery_formatted_text_logger', \Svenbluege\Component\Eventgallery\Site\Library\Common\FormattedTextLogger::class, true);
        Log::addLogger(
            array(
                'text_file' => self::COM_EVENTGALLERY_GOOGLEPHOTOS_LOGFILENAME,
                'logger' => 'com_eventgallery_formatted_text_logger'
            ),
            Log::ALL,
            'com_eventgallery'
        );
    }
}
