<?php

namespace Svenbluege\Component\Eventgallery\Site\Library\Connector;

use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\Exception\ExecutionFailureException;
use Joomla\Http\Response;
use OviDigital\JsObjectToJson\JsConverter;
use Svenbluege\Component\Eventgallery\Site\Library\Common\LogWorkaround;
use Svenbluege\Component\Eventgallery\Site\Library\Data\Exif;
use Svenbluege\Component\Eventgallery\Site\Library\File\File;

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

class GooglePhotosSharedPageConnector
{
    public static $cachebasedir;
    public static $cache_life = '86400'; //caching time, in seconds
    public static $requesttimeout = 5;

    /**
     * @return string
     */
    const COM_EVENTGALLERY_GOOGLEPHOTOS_SHAREDPAGE_LOGFILENAME = 'com_eventgallery_googlephotossharedpage.log.php';

    public static function doRequest($cachelifetime, $url, $hash, $data, $doCache)
    {
        self::initializeLogger();

        self::initCacheDirs();

        $cachefilename = self::$cachebasedir . $hash . '.txt';

        if ($doCache && file_exists($cachefilename) && (time() - filemtime($cachefilename) <= $cachelifetime)) {
            return $cachefilename;
        }

        //Log::add("Doing a request for $url", Log::INFO, 'com_eventgallery');
        /**
         * @var Response $result
         */

        try {

            $result = HttpFactory::getHttp()->get($url, ['Content-Type' => 'text/html'], self::$requesttimeout);

            $body = $result->body;

            if ($result->code < 300) {
                $fh = fopen($cachefilename, 'w') or die("can't open file $cachefilename");
                fwrite($fh, $body);
                fclose($fh);
            } else {
                if ($result->code == 400) {
                    Log::add("Request answered with HTTP 400 for URL: $url\n". $body, Log::ERROR, 'com_eventgallery');
                } else {
                    Log::add('Unable to connect to remote host. Make sure curl is available or allow_url_fopen=On and the server has access to the internet. url: ' . $url, Log::ERROR, 'com_eventgallery');
                }
            }

            // make sure we don't cause reload issues by loading elements too often. Max every minute should do.
            if (empty($body)) {
                Log::add("Invalid return result detected. The response body is empty for url $url. Please check the url for this album.", Log::ERROR, 'com_eventgallery');
                touch($cachefilename, time() - (self::$cache_life - 60)) ;
            }

            $result = NULL;

        } catch (\RuntimeException $e) {
            Log::add('Error for url '. $url .': '.$e->getMessage(), Log::ERROR, 'com_eventgallery');
        }

        return $cachefilename;
    }

    public static function initCacheDirs() {

        if (!is_dir(JPATH_CACHE)) {
            mkdir(JPATH_CACHE);
        }

        self::$cachebasedir = COM_EVENTGALLERY_GOOGLE_PHOTOS_SHAREDPAGE_CACHE_PATH;

        if (!is_dir(self::$cachebasedir)) {
            mkdir(self::$cachebasedir);
        }
    }



    /**
     * Updates the album with the database
     * @param $cachelifetime
     * @param $api_clientid
     * @param $api_secret
     * @param $refresh_token
     * @param $db DatabaseDriver
     * @param $albumId
     * @return void
     */
    public static function syncAlbum($cachelifetime, $db, $albumId, $url)
    {
        if (empty($url)) {
            return;
        }

        self::initCacheDirs();

        $serOBjectPath = self::$cachebasedir . $albumId .'.obj';

        if (file_exists($serOBjectPath) && (time() - filemtime($serOBjectPath) <= $cachelifetime)) {
            return null;
        }

        Log::add("Syncing the album $albumId.", Log::INFO, 'com_eventgallery');

        $hash_prefix = $albumId;

        $cachefilename = self::doRequest($cachelifetime, $url, $hash_prefix.'.album' , null, true);
        $htmlContent = @file_get_contents($cachefilename);

        $albumData = self::extractAlbumData($htmlContent);
        $photos = Array();
        $ordering = 0;

        foreach ($albumData['photos'] as $photoData) {

            $photo = [];

            $photo['type'] = $photoData['type'];

            $photo['baseurl'] = $photoData['baseUrl'];
            $photo['width'] = $photoData['width'];
            $photo['height'] = $photoData['height'];

            $photo['caption'] = '';
            $photo['title'] = '';
            $photo['creation_date'] = $photoData['creation_date'];
            $photo['folder'] = $albumId;
            $photo['file'] = md5($photoData['baseUrl']);
            $photo['googlephotos_filename'] = md5($photoData['baseUrl']);

            $exif = Array();

            $exif['fstop'] = '';
            $exif['focallength'] = '';
            $exif['exposuretime'] = '';
            $exif['model'] = '';
            $exif['iso'] = '';
            $exif['filename'] = md5($photoData['baseUrl']);

            $photo['exif'] = (object)$exif;
            $photo['published'] = 1;

            $photo['ordering'] = $ordering++;

            $photos[$photo['file']] = $photo;
            unset($photo);
        }

        $db->transactionStart();

        try {

            $query = $db->getQuery(true);
            $query->select('*')->from('#__eventgallery_file')
                ->where('folder='.$db->quote($albumId));
            $db->setQuery($query);
            $oldFiles = $db->loadObjectList();

            $oldFilesIndexedByFile = array_reduce($oldFiles, function ($carry, $oldFile) {
                $carry[$oldFile->file] = $oldFile;
                return $carry;
            }, []);

            $query = $db->getQuery(true);
            $query->delete('#__eventgallery_file')
                ->where('folder='.$db->quote($albumId));
            $db->setQuery($query);
            $db->execute();

            if (count($photos)>0) {
                foreach(array_chunk($photos, 100) as $photosChunk) {
                    self::addPhotosToDatabase($photosChunk, $albumId, $db, $oldFilesIndexedByFile);
                }
            }
            $db->transactionCommit();

            $c = 'just dummy content because we want to remember the last update time.';
            file_put_contents($serOBjectPath, $c);

        } catch (ExecutionFailureException $e) {
            Log::add('Catched database execution while updating Google Photos files. Error message: '. $e->getMessage(), Log::ERROR, 'com_eventgallery');
            $db->transactionRollback();
        }
    }

    private static function addPhotosToDatabase($photos, $albumId, $db, $oldFilesIndexedByFile) {

        $query = $db->getQuery(true);

        $query->insert($db->quoteName('#__eventgallery_file'))
            ->columns(
                       'id, 
                        folder,
                        file,
                        googlephotos_filename,
                        width,
                        height,
                        caption,
                        title,
                        googlephotos_baseurl,
                        url,
                        exif,
                        ordering,
                        ismainimage,
                        ismainimageonly,
                        type,
                        hits,
                        published,
                        userid,
                        modified,
                        created,
                        creation_date'
            );


        foreach ($photos as $photo) {
            $oldFile = isset($oldFilesIndexedByFile[$photo['file']]) ? $oldFilesIndexedByFile[$photo['file']] : null;
            $query->values(implode(',', array(
                isset($oldFile) ? $db->quote((int)$oldFile->id):0,
                $db->quote($albumId),
                $db->quote($photo['file']),
                $db->quote($photo['googlephotos_filename']),
                $db->quote($photo['width']),
                $db->quote($photo['height']),
                isset($oldFile) ? $db->quote($oldFile->title):$db->quote($photo['title']),
                isset($oldFile) ? $db->quote($oldFile->caption):$db->quote($photo['caption']),
                $db->quote($photo['baseurl']),
                $db->quote(''),
                $db->quote(json_encode($photo['exif'])),
                isset($oldFile) ? $db->quote((int)$oldFile->ordering):$db->quote((int)$photo['ordering']),
                isset($oldFile) ? $db->quote((int)$oldFile->ismainimage):0,
                isset($oldFile) ? $db->quote((int)$oldFile->ismainimageonly):0,
                $db->quote($photo['type']),
                isset($oldFile) ? $db->quote((int)$oldFile->hits):0,
                isset($oldFile) ? $db->quote((int)$oldFile->published):1,
                0,
                'now()',
                isset($oldFile) ? $db->quote($oldFile->created):'now()',
                isset($oldFile) && !empty($oldFile->creation_date) ? $db->quote($oldFile->creation_date): $db->quote($photo['creation_date'])
            )));

        }

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * @return void
     */
    private static function initializeLogger(): void
    {
        (new LogWorkaround())->registerLogger('com_eventgallery_formatted_text_logger', \Svenbluege\Component\Eventgallery\Site\Library\Common\FormattedTextLogger::class, true);
        Log::addLogger(
            array(
                'text_file' => self::COM_EVENTGALLERY_GOOGLEPHOTOS_SHAREDPAGE_LOGFILENAME,
                'logger' => 'com_eventgallery_formatted_text_logger'
            ),
            Log::ALL,
            'com_eventgallery'
        );
    }

    public static function extractAlbumData($text) {
        $pattern = '/AF_initDataCallback\((.*)\)/';
        $text = str_replace("<script ", "\n<script ", $text);
        preg_match_all($pattern, $text, $matches);

        $scriptDataArrays = array_reduce($matches[1],
            function ($carry, $match) {
                $carry[] = JsConverter::convertToArray($match);
                return $carry;
            },
            []);

        $album = [ "photos" => []];


        foreach($scriptDataArrays as $entry) {
            if ($entry['key'] != 'ds:1') continue;
            $photosDataArray = $entry['data'][1];
            foreach($photosDataArray as $photoDataArray) {
                $photo = [];
                $photo['baseUrl'] = $photoDataArray[1][0];
                $photo['width'] = $photoDataArray[1][1];
                $photo['height'] = $photoDataArray[1][2];
                $photo['type'] = isset($photoDataArray[15][76647426]) ? File::TYPE_VIDEO : File::TYPE_IMAGE;
                $photo['creation_date'] = date('YmdHis', intval($photoDataArray[2] / 1000));
                $album['photos'][] = $photo;
            }
            $albumDataArry = $entry['data'][3];
            $album['title'] = $albumDataArry[1];

        }

        return $album;
    }
}
