<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\Data;

defined('_JEXEC') or die;

use ImageInfo\ImageInfo;
use lsolesen\pel\PelJpeg;
use lsolesen\pel\PelIfd;
use lsolesen\pel\Pel;
use lsolesen\pel\PelTag;

class Exif
{
    public $model = null;
    public $focallength = null;
    public $focallength35mm = null;
    public $fstop = null;
    public $exposuretime = null;
    public $iso = null;
    public $creation_date = null;
    public $has90DegreeRotation = false;
    public $orientation = null;

    public $hasError = false;

    public function __construct($jsonStringData = null) {
        if (isset($jsonStringData)) {
            $data = json_decode($jsonStringData);
            if (json_last_error() == JSON_ERROR_NONE) {
                $this->model = isset($data->model)?$data->model : null;
                $this->focallength = isset($data->focallength)?$data->focallength : null;
                $this->focallength35mm = isset($data->focallength35mm)?$data->focallength35mm : null;
                $this->fstop = isset($data->fstop)?$data->fstop : null;
                $this->exposuretime = isset($data->exposuretime)?$data->exposuretime : null;
                $this->iso = isset($data->iso)?$data->iso : null;
                $this->creation_date = isset($data->creation_date)?$data->creation_date : null;
                $this->hasError = isset($data->hasError)?$data->hasError : false;
            }
        }
    }

    public function toJson() {
        return json_encode($this);
    }

    /**
     * @param $filename string Path to the file which contains the data
     * @param $autorotate boolean defines if we autorotate the image to get width/height correctly
     * @return Exif
     */
    public static function extract($filename) {

        $exif = new self();

        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        if ($ext == 'webp') {
            $errorLevel = error_reporting();
            error_reporting(E_ALL & ~E_WARNING);

            try {
                $imageInfo = new ImageInfo($filename);
                $exifData = $imageInfo->getEXIFData();
                if (!empty($exifData)) {
                        $exif->model = $exifData->get('Model') ?? null;
                        $exif->orientation = $exifData->getRaw('Orientation') ?? null;
                        if ($exifData->has('FocalLength')) {
                            $exif->focallength = sprintf('%.01f', self::exif_get_float($exifData->get('FocalLength')));
                        }
                        if ($exifData->has('FocalLengthIn35mmFilm')) {
                            $exif->focallength = sprintf('%.01f', self::exif_get_float($exifData->get('FocalLengthIn35mmFilm')));
                        }

                        $exif->fstop = $exifData->get('FNumber') ?? null;
                        $exif->exposuretime = $exifData->get('ExposureTime') ?? null;
                        $exif->iso = $exifData->get('ISOSpeedRatings') ?? null;
                        $exif->creation_date = $exifData->get('DateTimeOriginal') ? $exifData->get('DateTimeOriginal')->format('YmdHis') : null;
                    }
                }
            catch (\Exception|\Throwable $e) {
                $exif->hasError = true;
            }
            error_reporting($errorLevel);
        } elseif (function_exists('exif_read_data')) {

            $exifData = @exif_read_data($filename, 0, true);
            if (isset($exifData["IFD0"])) {
                if (isset($exifData["IFD0"]["Model"])) {
                    $exif->model = $exifData["IFD0"]["Model"];
                }

                if (isset($exifData['IFD0']['Orientation'])) {
                    $exif->orientation = $exifData['IFD0']['Orientation'];
                }
            }

            if (isset($exifData["EXIF"])) {
                if (isset($exifData["EXIF"]["ExposureTime"])) {
                    $exif->exposuretime = self::exif_format_exposure($exifData["EXIF"]["ExposureTime"]);
                }

                if(isset($exifData["EXIF"]["ISOSpeedRatings"])) {
                    $data = $exifData["EXIF"]["ISOSpeedRatings"];
                    if (is_array($data)) {
                        $exif->iso = $exifData["EXIF"]["ISOSpeedRatings"][0];
                    } else {
                        $exif->iso = $exifData["EXIF"]["ISOSpeedRatings"];
                    }
                }

                if (isset($exifData["EXIF"]["FocalLength"])) {
                    $focal_length = $exifData["EXIF"]["FocalLength"];
                    $exif->focallength = sprintf('%.01f', self::exif_get_float($focal_length));
                }

                if (isset($exifData["EXIF"]["FocalLengthIn35mmFilm"])) {
                    $focal_length = $exifData["EXIF"]["FocalLengthIn35mmFilm"];
                    $exif->focallength35mm = sprintf('%.01f', self::exif_get_float($focal_length));
                }

                if (isset($exifData["EXIF"]["DateTimeOriginal"])) {
                    $creation_date = $exifData["EXIF"]["DateTimeOriginal"];
                    if (!empty($creation_date)) {
                        $creation_date = str_replace(':', '', $creation_date);
                        $creation_date = str_replace(' ', '', $creation_date);
                        $exif->creation_date = $creation_date;
                    }
                }
            }

            if (isset($exifData["COMPUTED"]["ApertureFNumber"])) {
                $exif->fstop = str_replace('f/','', $exifData["COMPUTED"]["ApertureFNumber"]);
            }
        }

        if ($exif->orientation == 6 || $exif->orientation == 8) {
            $exif->has90DegreeRotation = true;
        }

        return $exif;
    }

    private static function exif_get_float($value) {
        $pos = strpos($value, '/');
        if ($pos === false) return (float) $value;
        $a = (float) substr($value, 0, $pos);
        $b = (float) substr($value, $pos+1);
        return ($b == 0) ? ($a) : ($a / $b);
    }

    private static function exif_format_exposure($value) {
        $v = explode('/', $value);
        try {
            if (count($v) == 2) {
                if ($v[0] / $v[1] < 1) {
                    return Pel::fmt('1/%d', $v[1] / $v[0]);
                } else {
                    return Pel::fmt('%d', $v[0] / $v[1]);
                }
            }
        } catch (\DivisionByZeroError $e) {
            $value = 0;
        }
        return $value;
    }


    /**
     * Converts a float into a fraction if the float is smaller than 1.
     *
     * @param $n
     * @param $tolerance
     * @return mixed|string
     */
    static function float2rat($n, $tolerance = 1.e-4) {
        $n = (float)$n;

        if ($n>=1 || $n<=0) {
            return $n;
        }
        $h1=1; $h2=0;
        $k1=0; $k2=1;
        $b = 1/$n;
        do {
            $b = 1/$b;
            $a = floor($b);
            $aux = $h1; $h1 = $a*$h1+$h2; $h2 = $aux;
            $aux = $k1; $k1 = $a*$k1+$k2; $k2 = $aux;
            $b = $b-$a;
        } while (abs($n-$h1/$k1) > $n*$tolerance);

        return "$h1/$k1";
    }

}

