<?php

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\Factory;
use Joomla\CMS\Factory;
use Svenbluege\Component\Eventgallery\Administrator\Table\OrderTable;
use Svenbluege\Component\Eventgallery\Site\Library\Cart;
use Svenbluege\Component\Eventgallery\Site\Library\Database\Sequence;
use Svenbluege\Component\Eventgallery\Site\Library\ImageLineitem;
use Svenbluege\Component\Eventgallery\Site\Library\Order;
use Svenbluege\Component\Eventgallery\Site\Library\OrderStatus;
use Svenbluege\Component\Eventgallery\Site\Library\ServiceLineitem;

defined('_JEXEC') or die();

/**
 * @extends AbstractFactory<OrderFactory>
 */
class OrderFactory extends AbstractFactory
{


    /**
     * @param Cart $cart
     * @return Order
     */
    public function createOrder($cart) {

        $db = $this->db;
        $data = $cart->_getInternalDataObject();
        // avoid getting the creation date of the cart as the order creation date.
        unset($data['created']);

        $uuid = str_replace('.','0',uniqid("", true));
        $uuid = base_convert($uuid,16,10);

        $token = uniqid("", true);

        $query = $db->getQuery(true);
        $query->insert("#__eventgallery_order");
        $query->columns($db->quoteName(array('id', 'token')));
        $query->values(implode(',', Array($db->quote($uuid), $db->quote($token))));
        $db->setQuery($query);
        $db->execute();


        $user = Factory::getUser();

        $data['id'] = $uuid;
        $data['token'] = $token;

        if (!$user->guest) {
            $data['userid'] = $user->id;
        }

        $lang = Factory::getLanguage();
        $langTag = $lang->getTag();
        $data['language'] = $langTag;

        /**
         * @var OrderTable $orderTable
         */
        $orderTable = $this->store($data, 'Order');
        /**
         * @var ImageLineitem $lineitem
         * @var ImageLineitemFactory $imageLineItemFactory
         */

        $imageLineItemFactory = ImageLineitemFactory::getInstance();

        foreach ($cart->getLineItems() as $lineitem) {
            $imageLineItemFactory->copyLineItem($orderTable->id, $lineitem);
        }

        /**
         * @var ServiceLineitem $lineitem
         * @var ServiceLineitemFactory $serviceLineItemFactory
         */
        $serviceLineItemFactory = ServiceLineitemFactory::getInstance();

        foreach ($cart->getServiceLineItems() as $lineitem) {
            $serviceLineItemFactory->copyLineItem($orderTable->id, $lineitem);
        }


        /**
         * @var OrderStatusFactory $orderstatusFactory
         */
        $orderstatusFactory = OrderStatusFactory::getInstance();

        /**
         * @var Order $order
         */
        $order = new Order($orderTable);
        $order->setOrderStatus($orderstatusFactory->getDefaultOrderStatus(OrderStatus::TYPE_ORDER));
        $order->setPaymentStatus($orderstatusFactory->getDefaultOrderStatus(OrderStatus::TYPE_PAYMENT));
        $order->setShippingStatus($orderstatusFactory->getDefaultOrderStatus(OrderStatus::TYPE_SHIPPING));
        $order->setDocumentNumber(Sequence::generateNewId());

        return $order;
    }

    public function getOrdersByUserId($userid) {
        $db = $this->db;

        $query = $db->getQuery(true);
        $query->select('*');
        $query->from("#__eventgallery_order");
        $query->where("userid=".$db->quote($userid));
        $query->order("created desc");

        $db->setQuery($query);
        $rows = $db->loadObjectList();
        $orders = array();
        foreach ($rows as $row) {
            array_push($orders, new Order($row));
        }

        return $orders;
    }

    /**
     * @param string $documentNo
     * @return Order
     */
    public function getOrdersByDocumentNumber($documentNo)
    {
        if ($documentNo<0) {
            return null;
        }

        $db = $this->db;

        $query = $db->getQuery(true);
        $query->select('*');
        $query->from("#__eventgallery_order");
        $query->where("documentno=".$db->quote($documentNo));


        $db->setQuery($query);
        $row = $db->loadObject();

        if ($row == null) {
            return null;
        }

        $order = new Order($row);

        return $order;
    }

    /**
     * Returns the order oject for a given ID
     *
     * @param $id
     * @return Order
     */
    public function getOrderById($id) {

        $db = $this->db;
        $query = $db->getQuery(true);

        $query->select('o.*');
        $query->from('#__eventgallery_order as o');
        $query->where('o.id = ' . $db->quote($id));
        $db->setQuery($query);

        $result = $db->loadObject();

        return $result == null ? null : new Order($result);
    }

    /**
     * @param $email
     * @return array
     */
    public function getOrdersByEmail($email) {
        $db = $this->db;
        $query = $db->getQuery(true);

        $query->select('o.id');
        $query->from('#__eventgallery_order as o');
        $query->where('o.email = ' . $db->quote($email));
        $db->setQuery($query);

        $orderIds = $db->loadColumn(0);

        $orders = [];
        foreach($orderIds as $orderId) {
            array_push($orders, $this->getOrderById($orderId));
        }

        return $orders;

    }


}
