<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Site\Library\File;
use DateTime;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Svenbluege\Component\Eventgallery\Administrator\Table\FileTable;
use Svenbluege\Component\Eventgallery\Site\Library\Data\Exif;
use Svenbluege\Component\Eventgallery\Site\Library\Database\Localizablestring;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FolderFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\ImageTypeSetFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\Folder;
use Svenbluege\Component\Eventgallery\Site\Library\Helper\SizeCalculator;
use Svenbluege\Component\Eventgallery\Site\Library\ImageTypeSet;
use Svenbluege\Component\Eventgallery\Site\Library\Interface\Image;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\FolderManager;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\OrderManager;

defined('_JEXEC') or die();


abstract class File implements Image
{
    const TYPE_VIDEO = 'video';
    const TYPE_IMAGE = 'image';
    /**
     * @var \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main
     */
    protected $config;

    /**
     * @var Exif
     */
    protected $exif;

    public $_blank_script_path = 'media/com_eventgallery/frontend/images/blank.gif';

    /**
     * @var string
     */
    protected $_filename = NULL;

    /**
     * @var string
     */
    protected $_foldername = NULL;

    /**
     * @var FileTable
     */
    protected $_file = NULL;

    /**
     * @var Folder
     */
    protected $_folder = NULL;

    protected $_ls_caption = NULL;

    protected $_ls_title = NULL;

    /**
     * @var ImageTypeSet
     */
    protected $_imagetypeset = NULL;

    protected $sizeCalculator = NULL;

    protected $_doLazyLoading = true;

    /**
     * creates the lineitem object. $dblineitem is the database object of this line item
     *
     * @param $object object
     */
    function __construct($object)
    {

        if (!is_object($object)) {
            throw new \InvalidArgumentException("Can't create File Object without a valid data object.");
        }

        $this->_file = $object;
        $this->_foldername = $object->folder;
        $this->_filename = $object->file;

        $this->exif = new Exif($this->_file->exif);

        /**
         * @var FolderFactory $folderFactory
         */
        $folderFactory = FolderFactory::getInstance();

        $this->_folder = $folderFactory->getFolder($object->folder);

        $this->_ls_title = new Localizablestring($this->_file->title);
        $this->_ls_caption = new Localizablestring($this->_file->caption);

        /**
         * @var ImageTypeSetFactory $imagetypesetFactory
         */
        $imagetypesetFactory = ImageTypeSetFactory::getInstance();

        if ($this->_file->imagetypesetid>0) {
            $imagetypeset = $imagetypesetFactory->getImageTypeSet($this->_file->imagetypesetid);
            if ($imagetypeset->isPublished()) {
                $this->_imagetypeset = $imagetypeset;
            }
        }

        $this->config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();
        $this->_doLazyLoading = $this->config->getImage()->doUseLazyLoadingForImages();

        if ($this->getWidth() && $this->getHeight()) {
            $this->sizeCalculator = new SizeCalculator($this->getWidth(), $this->getHeight(), COM_EVENTGALLERY_IMAGE_ORIGINAL_MAX_WIDTH, false);
        } else {
            $this->sizeCalculator = new SizeCalculator(1,1,1,false);
        }

    }

    /**
     * @return string
     */
    public function getFileName()
    {
        return $this->_filename;
    }

    /**
     * @return string
     */
    public function getFolderName() {
        return $this->_foldername;
    }

    /**
     * @return Folder
     */
    public function getFolder() {
        return $this->_folder;
    }

    /**
     * @return bool
     */
    public function isPublished()
    {
        return $this->getFolder()->isPublished() == 1 && $this->_file->published == 1;
    }

    /**
     * checks if the image has a title to show.
     * @param $showImageID
     * @param $showExif
     * @param $showImageTitle
     * @param $showImageCaption
     * @return bool
     */
    public function hasTitle($showImageID, $showExif, $showImageTitle, $showImageCaption)
    {
        if (!empty($this->getTitle($showImageID, $showExif, $showImageTitle, $showImageCaption))) {
            return true;
        }

        return false;
    }

    /**
     * returns the title of an image.
     * @param bool $showImageID
     * @param bool $showExif
     * @param $showImageTitle
     * @param $showImageCaption
     * @return string
     */
    public function getTitle($showImageID, $showExif, $showImageTitle, $showImageCaption)
    {
        return $this->getLightBoxTitle($showImageID, $showExif, $showImageTitle, $showImageCaption);
    }

    public function getHeight() {
        return $this->_file->height;
    }

    public function getWidth() {
        return $this->_file->width;
    }

    /**
     * @return int
     */
    public function getLightboxImageWidth() {
        $width = $this->sizeCalculator->getWidth();
        if ($this->getWidth() < $width) {
            return $this->getWidth();
        }
        return $width;
    }

    /**
     * @return int
     */
    public function  getLightboxImageHeight() {
        // just in case I'll forget: use the same logic for the height as for the width to avoid any issues.
        $width = $this->sizeCalculator->getWidth();
        if ($this->getWidth() < $width) {
            return $this->getHeight();
        }
        return $this->sizeCalculator->getHeight();
    }

    /**
     *  returns a title with the following format:
     *
     *   <span class="img-caption img-caption-part1">Foo</span>[<span class="img-caption img-caption-part1">Bar</span>][<span class="img-exif">EXIF</span>]
     *
     * @param bool $showImageID
     * @param bool $showExif
     * @param $showImageTitle
     * @param $showImageCaption
     * @return string
     */
    public function getLightBoxTitle($showImageID, $showExif, $showImageTitle, $showImageCaption)
    {

        $lightBoxTitle = "";

        $fileTitle = $this->getFileTitle();

        if ($showImageTitle && !empty($fileTitle)) {
            $lightBoxTitle .= '<span class="img-caption img-caption-part1">' . $fileTitle . '</span>';
        }

        $fileCaption = $this->getFileCaption();

        if ($showImageCaption && !empty($fileCaption)) {
            $lightBoxTitle .= '<span class="img-caption img-caption-part2">' . $fileCaption . '</span>';
        }

        if ($showExif) {
            $exifAsText = $this->getExifAsString();
            if (!empty($exifAsText)) {
                $lightBoxTitle .= '<span class="img-exif">';
                $lightBoxTitle .= $exifAsText;
                $lightBoxTitle .= "</span>";
            }
        }

        if ($showImageID) {
            $lightBoxTitle .=  '<span class="img-id">'.Text::_('COM_EVENTGALLERY_IMAGE_ID').' '.$this->getFileName().'</span>';

        }


        return $lightBoxTitle;
    }

    public function getThumbImgTag($width,  $height, $cssClass, $crop, $alternateThumbUrl, $showImageTitle, $showImageCaption) {
        $newWidth = $width;
        $newHeight = $height;

        if ($crop === false) {
            $newHeight = $this->getHeight()/$this->getWidth() * $width;
        }

        return '<img class="eg-img '.$cssClass.'" src="'.($alternateThumbUrl == null ? $this->getThumbUrl($width,$height, true) : $alternateThumbUrl).'" '.
            'style="width: '.$newWidth.'px; '.
            'height: '.$newHeight.'px; '.
            '" '.
            'alt="'.$this->getAltContent($showImageTitle, $showImageCaption).'" '.
            '/>';
    }

    /**
     * @param int $width
     * @param int $height
     * @param string $cssClass
     * @param bool|false $crop
     * @param string $customDataAttributes a string like "data-flickr-farm"
     * @param $showImageTitle
     * @param $showImageCaption
     * @return string
     */
    public function getLazyThumbImgTag($width,  $height, $cssClass, $crop, $customDataAttributes, $showImageTitle, $showImageCaption) {
        $cssClass .= ' eg-img';
        $imgTag = '<img '.
            'data-width="'.$this->getLightboxImageWidth().'" '.
            'data-height="'.$this->getLightboxImageHeight().'" '.
            $customDataAttributes.
            'data-src="' . htmlspecialchars($this->getThumbUrl($width, $height, true), ENT_NOQUOTES, "UTF-8") . '" ' .
            'src="' . htmlspecialchars($this->getImageUrl(null, null, true), ENT_NOQUOTES, "UTF-8") . '" ';

        if ($this->_doLazyLoading === true) {
            $imgTag .= 'class="eventgallery-lazyme eventgallery-lazyload-unloaded '.$cssClass.'" ';
            $imgTag .= 'loading="lazy" height="500" width="500"';
            $imgTag .= 'srcset="' . Uri::root() . $this->_blank_script_path . '" ';
        } else {
            $imgTag .= 'class="'.$cssClass.'" ';
            $imgTag .= 'srcset="'. htmlspecialchars($this->getThumbUrl($width, $height, true), ENT_NOQUOTES, "UTF-8") . '" ';
        }

        $imgTag.=
            'style="'.
                   'width: '.$width.'px; '.
                   'height: '.$height.'px; '.
                  '" ' .

            'alt="'.$this->getAltContent($showImageTitle, $showImageCaption).'" '.
            '/>';
        return $imgTag;
    }

    public function getImageSrcSet() {
        $sizes = [1024, 1600, 2048, 3600];
        $sourceSet = [];

        foreach ($sizes as $size) {
            if ($size > $this->getWidth()) {
                continue;
            }
            $sourceSet[] = $this->getThumbUrl($size, 0) . ' ' . $size . 'w';
        }

        return join(', ', $sourceSet);
    }
    public function getCartThumb($lineitem)
    {
        return '<a class="img-thumbnail thumbnail"
    						href="' . $this->getImageUrl(NULL, NULL, true) . '"
    						title="' . htmlentities($lineitem->getImageType()!=null?$lineitem->getImageType()->getDisplayName():"n/a", ENT_QUOTES, "UTF-8") . '"
    						data-title="' . rawurlencode($lineitem->getImageType()!=null?$lineitem->getImageType()->getDisplayName():"n/a") . '"
    						data-pid="'.$this->getId().'" data-width="'.$this->getLightboxImageWidth().'" data-height="'.$this->getLightboxImageHeight().'"
    						data-gid="cart"
    						data-lineitem-id="' . $lineitem->getId() . '"
    						data-eg-lightbox="cart"> ' . $this->getThumbImgTag(COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_CART_WIDTH, COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_CART_WIDTH, null, true, null, true, true) . '</a>';
    }

    /**
     * @param $lineitem AbstractImagelineitem
     */
    public function getMailThumbUrl($lineitem) {

        $config = Factory::getConfig();
        $sslmode = $config->get('force_ssl', 0) == 2 ? 1 : (2);
        /**
         * @var $orderMgr OrderManager
         */
        $orderMgr = OrderManager::getInstance();
        $order = $orderMgr->getOrderById($lineitem->getLineItemContainerId());
        // do not use  Route::_() here to avoid issues with protected frontend pages.
        $url = str_replace("/administrator", "", Uri::base() . "index.php?option=com_eventgallery&view=download&task=mailthumb&orderid=" . $order->getId() . "&lineitemid=" . $lineitem->getId() . "&token=" . $order->getToken());

        return $url;
    }

    /**
     * @param $lineitem AbstractImagelineitem
     */
    public function getOrderThumb($lineitem) {

        $url = $this->getMailThumbUrl($lineitem);
        return $this->getThumbImgTag(COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_CART_WIDTH, COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_CART_WIDTH, "", true, $url ,true, true);
    }



    public function getMiniCartThumb($lineitem)
    {
        return '<a class="img-thumbnail thumbnail"
    						href="' . $this->getImageUrl(NULL, NULL, true) . '"
    						title="' . htmlentities($lineitem->getImageType()!=null?$lineitem->getImageType()->getDisplayName():"n/a", ENT_QUOTES, "UTF-8") . '"
    						data-title="' . rawurlencode($lineitem->getImageType()!=null?$lineitem->getImageType()->getDisplayName():"n/a") . '"
    						data-pid="'.$this->getId().'" data-width="'.$this->getLightboxImageWidth().'" data-height="'.$this->getLightboxImageHeight().'"
    						data-gid="cart"
    						data-lineitem-id="' . $lineitem->getId() . '"
    						data-eg-lightbox="cart"> ' . $this->getThumbImgTag(COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_MINICART_WIDTH, COM_EVENTGALLERY_IMAGE_THUMBNAIL_IN_MINICART_WIDTH, null, true, null, true, true) . '</a>';
    }

    /**
     * returns the title of an image.
     */
    public function getPlainTextTitle($showImageTitle, $showImageCaption)
    {

        $title = $this->getFileTitle();
        if ($showImageTitle && !empty($title)) {
            return strip_tags($this->getFileTitle());
        }

        $caption = $this->getFileCaption();
        if ($showImageCaption && !empty($caption)) {
            return strip_tags($this->getFileCaption());
        }

        return "";
    }


    /**
     * counts a hit on this file.
     */
    public function countHit() {
        return;
    }

    /**
     * returns the number of hits for this file
     *
     * @return int
     */
    public function getHitCount() {
        if (isset($this->_file->hits)) {
            return $this->_file->hits;
        }
        return 0;
    }

    /**
     * returns the content for the alt attribute of an img tag.
     * @return string
     */
    public function getAltContent($showImageTitle, $showImageCaption) {


        $folderDisplayName = $this->getFolder()->getDisplayName();
        $title = $this->getPlainTextTitle($showImageTitle, $showImageCaption);

        $description = [];

        if (!empty($title)) {
            $description[] =  $title;
        }

        if (!empty($folderDisplayName)) {
            $description[] = $folderDisplayName;
        }

        $altContent = implode(' - ', $description);
        return htmlentities(strip_tags($altContent), ENT_QUOTES, "UTF-8");
    }

    /**
     * Returns the title of the image
     *
     * @param string $languageTag
     * @return string
     */
    public function getFileTitle($languageTag = null) {
        if (null == $this->_ls_title) {
            return "";
        }
        return $this->_ls_title->get($languageTag);
    }

    /**
     * Return the raw title without any language decoding magic.
     *
     * @return string
     */
    public function getRawFileTitle() {
        return $this->_file->title;
    }

    /**
     * Returns the title of the image
     *
     * @param string $languageTag
     * @return string
     */
    public function getFileCaption($languageTag = null) {
        if ($this->_ls_caption == null) {
            return "";
        }
        return $this->_ls_caption->get($languageTag);
    }

    /**
     * Return the raw caption without any language decoding magic.
     *
     * @return string
     */
    public function getRawFileCaption() {
        return $this->_file->caption;
    }

    /**
     * returns the id of the file
     * @return int
     */
    public function getId() {
        return $this->_file->id;
    }

    /**
     * Checks of the file has an url
     *
     * @return bool
     */
    public function hasUrl() {
        if (isset($this->_file->url) && !empty($this->_file->url)) {
            return true;
        }

        return false;
    }

    /**
     * return the url for this file
     *
     * @return string
     */
    public function getUrl() {
        if (!$this->hasUrl()) {
            return null;
        }
        return $this->_file->url;
    }

    /**
     * returns the creation date as a string with the format YYYYmmddHHiiss
     *
     * @return String
     */
    public function getCreationDateString() {
        return $this->_file->creation_date;
    }

    /**
     * @return DateTime|null
     */
    public function getCreationDate() {

        $dateStr = $this->getCreationDateString();
        if (empty($dateStr)) {
            return null;
        }

        $date = DateTime::createFromFormat('YmdHis', $dateStr);

        return $date;
    }

    /**
     * returns the ordering number of this file
     * @return int
     */
    public function getOrdering() {
        return $this->_file->ordering;
    }

    /**
     * Syncs this file with the database for example.
     */
    public function syncFile() {
        return FolderManager::$SYNC_STATUS_NOSYNC;
    }

    /**
     *
     * @return boolean
     */
    public function isMainImage() {
        return $this->_file->ismainimage;
    }

    public function isMainImageOnly() {
        return $this->_file->ismainimageonly;
    }

    public function getType() {
        return $this->_file->type;
    }

    public function isVideo() {
        return $this->getType() == File::TYPE_VIDEO;
    }

    public function isImage() {
        return !$this->isVideo();
    }

    public function getOriginalFile()
    {
        return file_get_contents($this->getOriginalImageUrl());
    }

    public function isCartable() {
        return $this->getFolder()->isCartable() && $this->isImage();
    }

    public function isShareable() {
        return $this->getFolder()->isShareable();
    }

    public function getOriginalFilename() {
        return $this->getFileName();
    }

    public function getExif()
    {
        return $this->exif;
    }

    public function __toString() {
        return $this->getId() . ' ' . $this->getFolderName() . "/" . $this->getFileName();
    }

    public function deleteImageFile() {
        throw new \InvalidArgumentException('unsupported method');
    }

    public function deleteThumbnails() {
        throw new \InvalidArgumentException('unsupported method');
    }

    /**
     * @return ImageTypeSet
     */
    public function getImageTypeSet()
    {
        if ($this->_imagetypeset != null) {
            return $this->_imagetypeset;
        }

        return $this->getFolder()->getImageTypeSet();
    }

    /**
     * @return string
     */
    public function getExifAsString()
    {
        $exifdata = [];
        if (!empty($this->getExif()->model)) $exifdata[] = $this->getExif()->model;

        $focallength = '';
        if (!empty($this->getExif()->focallength)) $focallength .= $this->getExif()->focallength . "mm";
        if (!empty($this->getExif()->focallength35mm)) $focallength .= ' (' . $this->getExif()->focallength35mm . "mm" . ')';
        $exifdata[] = $focallength;

        if (!empty($this->getExif()->fstop)) $exifdata[] = "f/" . $this->getExif()->fstop;
        if (!empty($this->getExif()->exposuretime)) $exifdata[] = $this->getExif()->exposuretime;
        if (!empty($this->getExif()->iso)) $exifdata[] = "ISO " . $this->getExif()->iso;

        return implode(', ', $exifdata);
    }

    public function createThumbnails() {

    }
}
