<?php

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\File;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\FlickrFolder;

defined('_JEXEC') or die();

class FlickrFile extends File
{
    /**
     * creates the lineitem object. $dblineitem is the database object of this line item
     *
     * @param object $object
     * @throws Exception
     */
    function __construct($object)
    {
        parent::__construct($object);
    }

    /**
     * @return int
     */
    public function getLightboxImageWidth() {
        return $this->getWidth();
    }

    /**
     * @return int
     */
    public function getLightboxImageHeight() {
        return $this->getHeight();
    }

    public function getFarmId() {
        return $this->_file->flickr_farm;
    }

    public function getServerId() {
        return $this->_file->flickr_server;
    }

    public function getSecret() {
        return $this->_file->flickr_secret;
    }

    public function getSecretO() {
        return $this->_file->flickr_secret_o;
    }

    public function getSecretH() {
        return $this->_file->flickr_secret_h;
    }

    public function getSecretK() {
        return $this->_file->flickr_secret_k;
    }


    public function getLazyThumbImgTag($width, $height, $cssClass, $crop, $customDataAttributes, $showImageTitle, $showImageCaption) {


        $customDataAttributes = ' data-server="'.$this->getServerId().'"';
        $customDataAttributes .= ' data-secret="'.$this->getSecret().'"';
        $customDataAttributes .= ' data-secret_o="'. ((COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE > 2048) ? $this->getSecretO()  : '').'"';
        $customDataAttributes .= ' data-secret_h="'. ((COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE >= 2048) ? $this->getSecretH() : '').'"';
        $customDataAttributes .= ' data-secret_k="'. ((COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE >= 1600) ? $this->getSecretK() : '').'"';
        $customDataAttributes .= ' data-id="'.$this->getFileName().'"';
        $customDataAttributes .= ' data-type="'.$this->getType().'"';
        $customDataAttributes .= ' ';

        return parent::getLazyThumbImgTag($width, $height, $cssClass, $crop, $customDataAttributes, $showImageTitle, $showImageCaption);
    }

    public function getImageUrl($width=104,  $height=104, $fullsize=false, $larger=false, $relative=false) {
        if ($fullsize) {
            return $this->getThumbUrl(1600, 1600);
        } else {
            return $this->getThumbUrl($width, $height);
        }
    }

    public function getVideoUrl()
    {
        /**
         * @var FlickrFolder $folder
         */
        $folder = $this->getFolder();
        $flickrAccount = $folder->getFlickrAccount();
        if (!isset($flickrAccount)) {
            return "";
        }
        $userid = $folder->getFlickrAccount()->getUserId();
        $origSecret = $this->getSecretO();
        $id = $this->getFileName();

        return "https://www.flickr.com/photos/$userid/{$id}/play/orig/{$origSecret}/";
    }

    public function getThumbUrl ($width=104, $height=104, $larger=true, $relative=false) {
        if ($width == 0) {
            $width = 104;
        }
        if ($height == 0) {
            $height = 104;
        }
        return $this->flickrUrlBuilder($this->getFlickrSizeCode($width, $height));
    }

    public function getOriginalImageUrl($forceOriginalFile = false) {
		if (!$forceOriginalFile)
		{
			if (COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE <= 1024) return $this->flickrUrlBuilder('b');
			if (COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE <= 1600) return $this->flickrUrlBuilder('h');
			if (COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE <= 2048) return $this->flickrUrlBuilder('k');
		}
    	return $this->flickrUrlBuilder('o');
    }

    public function getSharingImageUrl() {
	    return $this->getOriginalImageUrl();
    }

    /**
     *
     * s    klein, quadratisch, 75 x 75
     * q    large square 150x150
     * t    Thumbnail, 100 an der Längsseite
     * m    klein, 240 an der Längsseite
     * n    small, 320 on longest side
     * -    mittel, 500 an der Längsseite
     * z    mittel 640, 640 an der längsten Seite
     * c    mittel 800, 800 an der längsten Seite†
     * b    groß, 1024 an der längsten Seite*
     * h    groß mit 1600 Pixel, 1600 an längster Seite†
     * k    groß mit 2048 Pixel, 2048 an längster Seite†
     * o    Originalbild, entweder JPG, GIF oder PNG, je nach Quellformat
     *
     * @param $size
     * @return string
     */
    private function flickrUrlBuilder($sizeCode) {
        switch ($sizeCode) {
            case "h":
                $secret = $this->getSecretH();
                break;
            case "k":
                $secret = $this->getSecretK();
                break;
            case "o":
                $secret = $this->getSecretO();
                break;
            default:
                $secret = $this->getSecret();
        }

        $size = $sizeCode == '-' ? '' : '_'.$sizeCode;

        if ($this->isVideo()) {
            return "https://live.staticflickr.com/{$this->getServerId()}/{$this->getFileName()}_{$secret}_{$sizeCode}.jpg";
        } else {
            return 'https://live.staticflickr.com/'.$this->getServerId().'/'.$this->getFileName().'_'.$secret.$size.'.'.$this->getFileExtension();
        }
    }

    private function getFlickrSizeCode($width, $height) {
        $sizes = array(100 => 't', 240 => 'm', 320 => 'n', 500 => '-', 640 => 'z', 800 => 'c', 1024 => 'b');

        $longSideSize = $width;
        $originalLongSideSize = $this->getWidth();


        if ($this->getHeight() > 0) {
            if ($height > $width) {
                $longSideSize = $height;
                $originalLongSideSize = $this->getHeight();
            }

            if ($height == $width) {
                $ratio = $this->getWidth() / $this->getHeight();
                if ($ratio < 1) {
                    // landscape
                    $longSideSize = $width / $ratio;
                } else {
                    //portait
                    $longSideSize = $width * $ratio;
                }
                if ($longSideSize > 2048) {
                    $longSideSize = 2048;
                }
            }
        }



	    if ($originalLongSideSize >= 1600 && COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE >= 1600 && $this->getSecretH() != "") {
		    $sizes[1600] = 'h';
	    }

	    if ($originalLongSideSize >= 2048 && COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE >= 2048 && $this->getSecretK() != "") {
		    $sizes[2048] = 'k';
	    }

	    foreach($sizes as $size=>$sizeCode) {
            if ($size >= $longSideSize) {
                return $sizeCode;
            }
        }

	    if (COM_EVENTGALLERY_FLICKR_MAX_IMAGE_SIZE > 2048) {
		    return 'o';
	    }

        return $sizeCode;
    }

    private function getFileExtension() {
        $ext = $this->_file->flickr_originalformat;
        return !empty($ext)? $ext : 'jpg';
    }

}
