<?php

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\File;

use FilesystemIterator;
use Joomla\CMS\Factory;
use Joomla\CMS\Filter\InputFilter;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Uri\Uri;
use Psr\Log\AbstractLogger;
use Svenbluege\Component\Eventgallery\Administrator\Table\FileTable;
use Svenbluege\Component\Eventgallery\Site\Library\Common\ImageProcessor;
use Svenbluege\Component\Eventgallery\Site\Library\Common\MediaHelper;
use Svenbluege\Component\Eventgallery\Site\Library\Common\Video;
use Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main;
use Svenbluege\Component\Eventgallery\Site\Library\Data\Exif;
use Svenbluege\Component\Eventgallery\Site\Library\Exception\UnsupportedFileExtensionException;
use Svenbluege\Component\Eventgallery\Site\Library\Helper\SizeSet;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\FolderManager;

defined('_JEXEC') or die();

class LocalFile extends \Svenbluege\Component\Eventgallery\Site\Library\File\File
{

    protected $_image_script_path = 'components/com_eventgallery/helpers/image.php?';

    /**
     * creates the lineitem object. $dblineitem is the database object of this line item
     *
     * @param object $object
     * @throws Exception
     */
    function __construct($object)
    {

        parent::__construct($object);

        if ($this->config->getImage()->doUseLegacyImageRendering()) {
            $this->_image_script_path = "index.php?option=com_eventgallery&view=resizeimage";
        }

        // this is necessary to avoid an exception while running in CLI mode
        if (array_key_exists('REQUEST_METHOD', $_SERVER)) {
            $currentApplicationName = Factory::getApplication()->getName();

            if ($currentApplicationName == 'administrator') {
                $this->_image_script_path .= '&site=1';
            }
        }
    }

    public function getImageUrl($width=104,  $height=104, $fullsize=false, $larger=false, $relative=false) {
        if ($fullsize) {
            return $this->createThumbnailUrl(COM_EVENTGALLERY_IMAGE_ORIGINAL_MAX_WIDTH, $relative);
        } else {
            $sizeSet = new SizeSet();
            $width = $sizeSet->getSizeCode($width, $height, $this->getWidth(), $this->getHeight());
            return $this->createThumbnailUrl($width, $relative);
        }
    }

    public function getVideoUrl()
    {
        $config = Main::getInstance();
        return Uri::root() . COM_EVENTGALLERY_IMAGE_FOLDER_WEB_PATH .'/'. urlencode($this->getFolderName()) . '/' . $this->getFileName() . ($config->getImage()->hasBust()?'?'.$config->getImage()->getBust():'');
    }
    public function getThumbUrl ($width=104, $height=104, $larger=true, $relative=false) {
        $sizeSet = new SizeSet();
        $width = $sizeSet->getSizeCode($width, $height, $this->getWidth(), $this->getHeight());
        return $this->createThumbnailUrl($width, $relative);
    }

    public function getOriginalImageUrl($forceOriginalFile = false) {

    	return Uri::base().substr( Route::_('index.php?option=com_eventgallery&view=download&folder='.$this->getFolderName().'&file='.urlencode($this->getFileName()) ), strlen(Uri::base(true)) + 1);

    }

    public function getSharingImageUrl() {

        return Uri::base().substr( Route::_('index.php?option=com_eventgallery&is_for_sharing=true&view=download&&folder='.$this->getFolderName().'&file='.urlencode($this->getFileName()) ), strlen(Uri::base(true)) + 1);

    }

    /**
     * increases the hit counter in the database
     */
    public function countHit() {
        /**
         * @var FileTable $table
         */
        $table = Factory::getApplication()->bootComponent('com_eventgallery')->getMVCFactory()->createTable('File', 'Administrator');
        $table->hit($this->_file->id);
    }

    public function syncFile() {
        $folderpath = COM_EVENTGALLERY_IMAGE_FOLDER_PATH.$this->getFolderName();
        self::updateMetadata($folderpath.DIRECTORY_SEPARATOR.$this->getFileName(), $this->getFolderName(), $this->getFileName());

        return FolderManager::$SYNC_STATUS_SYNC;
    }

    /**
     * upaded meta information
     * @param $path
     * @param $foldername
     * @param $filename
     */
    public static function updateMetadata($path, $foldername, $filename) {
        $config = Main::getInstance();

        if (!in_array(strtolower(pathinfo($filename, PATHINFO_EXTENSION)), COM_EVENTGALLERY_ALLOWED_FILE_EXTENSIONS)) {
            $extension = pathinfo($filename, PATHINFO_EXTENSION);
            throw new UnsupportedFileExtensionException("Extension $extension not in in allowed extension list " . join(', ', COM_EVENTGALLERY_ALLOWED_FILE_EXTENSIONS));
        }

        $mediaHelper = new MediaHelper();
        $mime = $mediaHelper->getMimeType($path, true);

        if (empty($mime) || !\in_array($mime, COM_EVENTGALLERY_ALLOWED_MIME_TYPES)) {
            throw new UnsupportedFileExtensionException("MimeType $mime not in in allowed list " . join(', ', COM_EVENTGALLERY_ALLOWED_MIME_TYPES));
        }

        /** @noinspection PhpUnusedLocalVariableInspection */
        @list($width, $height, $httpContentType, $attr) = getimagesize($path, $info);

        $type = File::TYPE_IMAGE;
        if (strpos($mime, 'video')>-1) {
            $type = File::TYPE_VIDEO;
            $dimension = Video::getDimension($path);
            $width = $dimension->getWidth();
            $height = $dimension->getHeight();
        }

        $creation_date = "";
        $title = "";
        $caption = "";

        if (isset($info["APP13"]) && function_exists("iptcparse")) {
            $iptc = iptcparse($info["APP13"]);
            if (is_array($iptc)) {
                if (isset($iptc[COM_EVENTGALLERY_IMAGE_METADATA_IPTC_FIELD_TITLE])) {
                    $title = $iptc[COM_EVENTGALLERY_IMAGE_METADATA_IPTC_FIELD_TITLE][0];
                }

                if (isset($iptc["2#055"])) {
                    $creation_date = $iptc["2#055"][0];
                    if (isset($iptc["2#060"])) {
                        $creation_date .= $iptc["2#060"][0];
                    }
                }

                if (isset($iptc[COM_EVENTGALLERY_IMAGE_METADATA_IPTC_FIELD_CAPTION])) {
                    $caption = $iptc[COM_EVENTGALLERY_IMAGE_METADATA_IPTC_FIELD_CAPTION][0];
                }
            }
        }


        $exif = Exif::extract($path);

        if ($config->getImage()->doAutoRotate() && $exif->has90DegreeRotation) {
            $tempWidth = $width;
            $width = $height;
            $height = $tempWidth;
        }

	    if (empty($creation_date)) {
            $creation_date = $exif->creation_date;
        }

        if(empty($creation_date)) {
            $file_creation_date = filectime($path);
            $creation_date = date('YmdHis', $file_creation_date);
        }

        // do some filtering for the content. We do not allow HTML in here.
        $filter = InputFilter::getInstance();
        $title = $filter->clean($title, 'html');
        $caption = $filter->clean($caption, 'html');
        $creation_date = $filter->clean($creation_date, 'html');


        $use_iptc_data = $config->getImage()->doUseIPTCData();
        $override_with_iptc_data = $config->getImage()->doOverwriteWithIPTCData();

        LocalFile::storeMetadata($foldername, $filename, $width, $height, $exif->toJson(), $creation_date, $use_iptc_data, $override_with_iptc_data, $title, $caption, $type);
    }

    static public function storeMetadata($foldername, $filename, $width, $height, $exifJson, $creation_date, $use_iptc_data, $override_with_iptc_data, $iptcTitle, $iptcCaption, $type) {
        $db = Factory::getDbo();

        $query = $db->getQuery(true);
        $query->update("#__eventgallery_file");
        $query->set("width=".$db->quote($width));
        $query->set("height=".$db->quote($height));
        $query->set("exif=".$db->quote($exifJson));
        $query->set("type=".$db->quote($type));
        $query->where('folder='.$db->quote($foldername));
        $query->where('file='.$db->quote($filename));
        $db->setQuery($query);
        $db->execute();

        if (!empty($creation_date)) {
            $query = $db->getQuery(true);
            $query->update("#__eventgallery_file");
            $query->set("creation_date=".$db->quote($creation_date));
            $query->where('folder=' . $db->quote($foldername));
            $query->where('file=' . $db->quote($filename));
            if ($override_with_iptc_data == false) {
                $query->where("(creation_date='' OR creation_date IS NULL)");
            }
            $db->setQuery($query);
            $db->execute();
        }

        if ($use_iptc_data && !empty($iptcCaption)) {
            $query = $db->getQuery(true);
            $query->update("#__eventgallery_file");
            $query->set("caption=" . $db->quote($iptcCaption));
            $query->where('folder=' . $db->quote($foldername));
            $query->where('file=' . $db->quote($filename));
            if ($override_with_iptc_data == false) {
                $query->where("(caption='' OR caption IS NULL)");
            }
            $db->setQuery($query);
            $db->execute();
        }

        if ($use_iptc_data && !empty($iptcTitle)) {
            $query = $db->getQuery(true);
            $query->update("#__eventgallery_file");
            $query->set("title=" . $db->quote($iptcTitle));
            $query->where('folder=' . $db->quote($foldername));
            $query->where('file=' . $db->quote($filename));
            if ($override_with_iptc_data == false) {
                $query->where("(title='' OR title IS NULL)");
            }
            $db->setQuery($query);
            $db->execute();
        }
    }

    public function createThumbnails() {
        $sizeSet = new SizeSet();
        $availableSizes = array_unique($sizeSet->availableSizes);

        foreach($availableSizes as $availableSize) {
            try {
                ImageProcessor::createThumbnail($this->getFolderName(), $this->getFileName(), $availableSize, true, true, null, true, true);
            } catch (UnsupportedFileExtensionException $e){

            }
        }

        return [$availableSizes];
    }

    public function getOriginalFile()
    {
        $filename = $this->getPathToOriginalFile();
        // try the path to a possible original file
        $basename = COM_EVENTGALLERY_IMAGE_FOLDER_PATH . $this->getFolderName() . DIRECTORY_SEPARATOR;
        $fullFilename = $basename. COM_EVENTGALLERY_IMAGE_ORIGINAL_SUBFOLDER . DIRECTORY_SEPARATOR . $this->getFileName();

        if (file_exists($fullFilename)) {
            $filename = $fullFilename;
        }

        return file_get_contents($filename);

    }

    /**
     * Deletes the image file
     */
    public function deleteImageFile() {
        $path= COM_EVENTGALLERY_IMAGE_FOLDER_PATH . \Joomla\Filesystem\File::makeSafe($this->getFolderName()).DIRECTORY_SEPARATOR;
        $filename=\Joomla\Filesystem\File::makeSafe($this->getFileName());
        $file = $path.$filename;

        $cacheFolder = COM_EVENTGALLERY_IMAGE_CACHE_PATH . \Joomla\Filesystem\File::makeSafe($this->getFolderName()).DIRECTORY_SEPARATOR;

        if (file_exists($cacheFolder)) {
            $fileIterator = new FilesystemIterator($cacheFolder, FilesystemIterator::SKIP_DOTS);
            while ($fileIterator->valid()) {
                $cacheFilename = $fileIterator->getFilename();
                if (preg_match("/(mainimage_)?nocrop_[\d]{2,4}_{$this->getFileName()}$/", $cacheFilename)) {
                    unlink($fileIterator->getPathname());
                }
                $fileIterator->next();
            }
        }

        if (file_exists($file) && !is_dir($file)) {
            if (!unlink($file)) {
                return false;
            }
        }
        return true;
    }

    public function deleteThumbnails()
    {
        $sizeSet = new SizeSet();
        $availableSizes = array_unique($sizeSet->availableSizes);

        foreach($availableSizes as $availableSize) {
            $image_thumb_file = ImageProcessor::calculateCacheThumbnailName($availableSize, true, $this->getFileName(), $this->getFolderName(), $this->isMainImage());
            if (file_exists($image_thumb_file)) {
                unlink($image_thumb_file);
            }
        }
    }

    /**
     * @param int $width
     * @param boolean $relative
     * @return string
     */
    protected function createThumbnailUrl($width, $relative)
    {
        $config = Main::getInstance();

        if ($config->getImage()->doUsePrecalculatedThumbnailsForLocalFiles()) {
            $sizeSet = new SizeSet();
            $saveAsSize = $sizeSet->getMatchingSize($width);
            $image_thumb_file = ImageProcessor::calculateCacheThumbnailName($saveAsSize, true, $this->getFileName(), $this->getFolderName(), $this->isMainImage());

            return Uri::root($relative) . COM_EVENTGALLERY_IMAGE_CACHE_WEB_PATH . '/' . urlencode($this->getFolderName()) . '/' . basename($image_thumb_file) . ($config->getImage()->hasBust()?'?'.$config->getImage()->getBust():'');
        }

        return Uri::root($relative) . $this->_image_script_path . "&width=" . $width . "&folder=" . $this->getFolderName() . "&file=" . urlencode($this->getFileName()) . ($config->getImage()->hasBust()?'&'.$config->getImage()->getBust():'');
    }

    /**
     * Get the path to the uploaded files in the images folder
     * @return string
     */
    private function getPathToOriginalFile(): string
    {
        $basename = COM_EVENTGALLERY_IMAGE_FOLDER_PATH . $this->getFolderName() . DIRECTORY_SEPARATOR;
        $filepath = $basename . $this->getFileName();
        return $filepath;
    }
}
