<?php

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\File;
use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Uri\Uri;
use Svenbluege\Component\Eventgallery\Administrator\Table\FileTable;
use Svenbluege\Component\Eventgallery\Site\Library\Common\ImageProcessor;
use Svenbluege\Component\Eventgallery\Site\Library\Common\LogWorkaround;
use Svenbluege\Component\Eventgallery\Site\Library\Common\S3Client;
use Svenbluege\Component\Eventgallery\Site\Library\Common\Video;
use Svenbluege\Component\Eventgallery\Site\Library\Data\Exif;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\WatermarkFactory;
use Svenbluege\Component\Eventgallery\Site\Library\File\File;
use Svenbluege\Component\Eventgallery\Site\Library\Helper\SizeSet;
use Svenbluege\Component\Eventgallery\Site\Library\Manager\FolderManager;
use Svenbluege\Component\Eventgallery\Site\Library\Watermark;

defined('_JEXEC') or die();



class S3File extends File
{
    private $expriationTime = '+10 minutes';

    /**
     * creates the lineitem object. $dblineitem is the database object of this line item
     *
     * @param object $object
     * @throws Exception
     */
    function __construct($object)
    {
        parent::__construct($object);

        $this->config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();

        if ($this->_file->width <= 0) {
            $this->_file->width = 1000;
        }
        if ($this->_file->height <= 0) {
            $this->_file->height = 1000;
        }

    }

    public function getImageUrl($width=104,  $height=104, $fullsize=false, $larger=false, $relative=false) {
        $config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();
        $bust = $config->getImage()->hasBust()?'?'.$config->getImage()->getBust():'';
        $s3client = S3Client::getInstance();
        if ($fullsize) {
            return $s3client->getURL($s3client->getEndpoint(), $s3client->getBucketForThumbnails(), $this->calculateS3Key(COM_EVENTGALLERY_IMAGE_ORIGINAL_MAX_WIDTH), true) . $bust;
        } else {
            $width = $this->getSizeCode($width, $height);
            return $s3client->getURL($s3client->getEndpoint(), $s3client->getBucketForThumbnails(), $this->calculateS3Key($width), true) . $bust;
        }
    }

    public function getVideoUrl()
    {
        $config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();
        $bust = $config->getImage()->hasBust()?'?'.$config->getImage()->getBust():'';
        $s3client = S3Client::getInstance();
        return $s3client->getURL($s3client->getEndpoint(), $s3client->getBucketForThumbnails(), $this->getFolderName() . "/" . $this->getFileName(), true) . $bust;
    }

    public function getThumbUrl ($width=104, $height=104, $larger=true, $relative=false) {
        $config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();
        $bust = $config->getImage()->hasBust()?'?'.$config->getImage()->getBust():'';
        $s3client = S3Client::getInstance();
        $width = $this->getSizeCode($width, $height);
        return $s3client->getURL($s3client->getEndpoint(), $s3client->getBucketForThumbnails(), $this->calculateS3Key($width), true) . $bust;
    }

    public function getOriginalImageUrl($forceOriginalFile = false) {

        return Uri::base().substr( Route::_('index.php?option=com_eventgallery&view=download&folder='.$this->getFolderName().'&file='.urlencode($this->getFileName()) ), strlen(Uri::base(true)) + 1);

    }

    public function getSharingImageUrl() {

        return Uri::base().substr( Route::_('index.php?option=com_eventgallery&is_for_sharing=true&view=download&folder='.$this->getFolderName().'&file='.urlencode($this->getFileName()) ), strlen(Uri::base(true)) + 1);

    }

    private function getSizeCode($width, $height) {
        $sizeSet = new SizeSet();
        return $sizeSet->getSizeCode($width, $height, $this->getWidth(), $this->getHeight());
    }

    public function getETag() {
        return $this->_file->s3_etag;
    }

    /**
     * returns an array(s3key => etag)
     *
     * @return array
     */
    public function getETagForThumbnails() {
        return json_decode($this->_file->s3_etag_thumbnails, true);
    }

    /**
     * increases the hit counter in the database
     */
    public function countHit() {
        /**
         * @var FileTable $table
         */
        $table = Factory::getApplication()->bootComponent('com_eventgallery')->getMVCFactory()->createTable('File', 'Administrator');
        $table->hit($this->_file->id);
    }

    public function syncFile() {

        $s3client = S3Client::getInstance();
        $key = $this->getFolderName() . "/" . $this->getFileName();

        $config   = Factory::getConfig();

        $tempFileName = tempnam($config->get('tmp_path'), 'eg');
        $s3file = $s3client->getObjectToFile($s3client->getBucketForOriginalImages(), $key, $tempFileName);

        $this->syncFileData($s3file, $tempFileName);

        if (is_file($tempFileName)) {
            //chown($tempFileName,666);
            unlink($tempFileName);
        }
        unset($s3file);
        gc_collect_cycles();

        return FolderManager::$SYNC_STATUS_SYNC;

    }

    /**
     * Create Thumbnails using the local server and upload them to S3
     *
     *
     * @return array
     */
    public function createThumbnails() {
        (new LogWorkaround())->registerLogger('com_eventgallery_formatted_text_logger', \Svenbluege\Component\Eventgallery\Site\Library\Common\FormattedTextLogger::class, true);
        Log::addLogger(
            array(
                'text_file' => 'com_eventgallery_s3.log.php',
                'logger' => 'com_eventgallery_formatted_text_logger'
            ),
            Log::ALL,
            'com_eventgallery'
        );

        $time_start = microtime(true);

        $s3client = S3Client::getInstance();
        $sizeSet = new SizeSet();
        $availableSizes = array_unique($sizeSet->availableSizes);
        $doWatermarking = true;
        $doSharping = true;

        $folderObject = $this->getFolder();
        $watermark = $folderObject->getWatermark();
        // load default watermark
        if (null == $watermark || !$watermark->isPublished()) {
            /**
             * @var WatermarkFactory $watermarkFactory
             * @var Watermark $watermark
             */
            $watermarkFactory = WatermarkFactory::getInstance();
            $watermark = $watermarkFactory->getDefaultWatermark();
        }

        $tempFileName = JPATH_CACHE . '/' . $this->getFileName();
        $s3file = $s3client->getObjectToFile($s3client->getBucketForOriginalImages(), $this->getFolderName() . "/" . $this->getFileName(), $tempFileName);

        $originalFileETag = $s3file->getCleanETag();

        if ($this->isVideo()) {
            $cachedOriginalFile = $tempFileName;
            $cachedOriginalFileThumbnail = $tempFileName . '_video_thumb.jpg';
            Video::extractImage($this, $cachedOriginalFile, $cachedOriginalFileThumbnail);
            $tempFileName = $cachedOriginalFileThumbnail;

            // create a readable copy in the public S3 bucket
            $thumbnail = $s3client->putObjectFile($s3client->getBucketForThumbnails(),
                $this->getFolderName() . "/" . $this->getFileName(),
                $cachedOriginalFile,
                S3Client::ACL_PUBLIC_READ
            );
        }


        $thumbnailETags = array();

        $joomlaConfig   = Factory::getApplication()->getConfig();

        foreach($availableSizes as $size) {
            $image_thumb_file = ImageProcessor::createThumbnailToFile($tempFileName, null, $this->config, $size, $doWatermarking, $watermark, $doSharping, false);

            $key = $this->calculateS3Key($size);

            $thumbnail = $s3client->putObjectFile($s3client->getBucketForThumbnails(),
                $key,
                $image_thumb_file,
                S3Client::ACL_PUBLIC_READ
            );

            if ($thumbnail) {
                $thumbnailETags[$key] = $thumbnail->getCleanETag();
            } else {
                Log::add('Failed to upload thumbnail '.$this->getFolderName(). '/'. $this->getFileName() . " " . $key, Log::ERROR, 'com_eventgallery');
            }

            // call this to clear the filenhandler. Otherwise we can't delete the temp file
            gc_collect_cycles();
            unlink($image_thumb_file);
        }

        if ($this->isVideo()) {
            LocalFile::updateMetadata($cachedOriginalFile, $this->getFolderName(),$this->getFileName());
            unlink($cachedOriginalFile);
            unlink($cachedOriginalFileThumbnail);
        } else {
            LocalFile::updateMetadata($tempFileName, $this->getFolderName(),$this->getFileName());
            unlink($tempFileName);
        }

        unset($s3file);
        gc_collect_cycles();

        $this->saveETags($originalFileETag, $thumbnailETags);

        $time_end = microtime(true);

        $execution_time = ($time_end - $time_start);

        Log::add('processing file '.$this->getFolderName(). '/'. $this->getFileName() . " in $execution_time s.", Log::INFO, 'com_eventgallery');
        return [$availableSizes];
    }

    /**
     * @param \Svenbluege\Component\Eventgallery\Site\Library\Common\S3File $s3file
     * @param string $tempFileName
     */
    public function syncFileData(\Svenbluege\Component\Eventgallery\Site\Library\Common\S3File $s3file, string $tempFileName): void
    {
        $db = Factory::getDbo();

        $etag = $s3file->getCleanETag();

        LocalFile::updateMetadata($tempFileName, $this->getFolderName(), $this->getFileName());

        if ($etag != $this->getETag()) {
            // update the etag since it has changed.
            $query = $db->getQuery(true);
            $query->update('#__eventgallery_file')
                ->set('s3_etag=' . $db->quote($etag))
                ->set('s3_etag_thumbnails=""')
                ->where('id=' . $db->quote($this->getId()));
            $db->setQuery($query);
            $db->execute();
        }
    }



    /**
     * remove double quoutes at the beginning and the end of an etag
     *
     * @param $etag
     * @return String
     */
    private function cleanETag($etag) {
        return str_replace("\"", "", $etag);
    }

    private function saveETags($originalFileETag, $thumbnailETags) {
        $db = Factory::getDbo();
        // update the etag since it has changed.
        $query = $db->getQuery(true);
        $query->update('#__eventgallery_file')
            ->set('s3_etag='. $db->quote($originalFileETag))
            ->set('s3_etag_thumbnails='. $db->quote(json_encode($thumbnailETags)))
            ->where('id=' . $db->quote($this->getId()));
        $db->setQuery($query);
        $db->execute();
    }

    public function calculateS3Key($size) {
        return $this->getFolderName() . '/'. $this->calculateS3KeyPart($size);
    }


    private function calculateS3KeyPart($size) {
        return 's'. $size . '/' . $this->getFileName();
    }

    public function getOriginalFile()
    {
        $s3client = S3Client::getInstance();
        $tempFileName = JPATH_CACHE . '/' . $this->getFileName();
        $image_thumb_file = null;
        $s3client->getObjectToFile(
            $s3client->getBucketForOriginalImages(),
            $this->getFolderName() . "/" . $this->getFileName(),
            $tempFileName);

        $fileContent = file_get_contents($tempFileName);

        unlink($tempFileName);

        return $fileContent;
    }

    /**
     * Deletes the image file
     */
    public function deleteImageFile() {
        $s3client = S3Client::getInstance();
        return $s3client->deleteObjectFile($s3client->getBucketForOriginalImages(), $this->getFolderName() . "/" . $this->getFileName());
    }

    public function deleteThumbnails()
    {
        $sizeSet = new SizeSet();
        $availableSizes = array_unique($sizeSet->availableSizes);
        $s3client = S3Client::getInstance();

        foreach($availableSizes as $availableSize) {
            $key = $this->calculateS3Key($availableSize);
            $s3client->deleteObjectFile($s3client->getBucketForOriginalImages(), $key);
        }
    }


}
