<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Site\Library;
use Svenbluege\Component\Eventgallery\Administrator\Table\CartTable;
use Svenbluege\Component\Eventgallery\Site\Library\Common\Money;
use Svenbluege\Component\Eventgallery\Site\Library\Database\DatabaseObject;
use Svenbluege\Component\Eventgallery\Site\Library\Enum\BasketType;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\AddressFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\ImageLineitemFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\ServiceLineitemFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Methods\PaymentMethod;
use Svenbluege\Component\Eventgallery\Site\Library\Methods\ShippingMethod;
use Svenbluege\Component\Eventgallery\Site\Library\Methods\SurchargeMethod;

defined('_JEXEC') or die();

/**
 * @property mixed cart
 */
abstract class LineitemContainer extends DatabaseObject
{
    /**
     * @var Address
     */
    protected $_billingaddress = null;
    /**
     * @var CartTable
     */
    protected $_lineitemcontainer = null;

    protected $_lineitemcontainer_id = null;
    /**
     * @var string
     */
    protected $_lineitemcontainer_table = null;
    /**
     * @var array
     */
    protected $_lineitems = null;

    /**
     * A hash map, key = imagetype, value = Array<ImageLineItem>
     *
     * @var array
     */
    protected $_lineitemImageTypeCache = null;

    /**
     * cache for the used image types
     *
     * @var array
     */
    protected $_usedImageTypesCache = null;

    /**
     * @var array
     */
    protected $_servicelineitems = null;
    /**
     * @var ShippingMethod
     */

    protected $_shippingaddress = null;
    /**
     * @var SurchargeMethod
     */

    protected $_user_id = null;

    /**
     * @var ImageLineitemFactory
     */
    protected $_imageLineItemFactory;

    /**
     * @var ServiceLineitemFactory
     */
    protected $_serviceLineItemFactory;

    public function __construct()
    {
        $this->_imageLineItemFactory = ImageLineitemFactory::getInstance();
        $this->_serviceLineItemFactory = ServiceLineitemFactory::getInstance();
        parent::__construct();
    }

    /**
     * Use this method never in your source code. This is only for managers.
     *
     * @return array
     */
    public function _getInternalDataObject()
    {
        return get_object_vars($this->_lineitemcontainer);
    }

    function deleteLineItem($lineItemId)
    {
        $this->deleteLineItems([$lineItemId]);
    }

    /**
     * @param array $lineItemIds
     */
    function deleteLineItems($lineItemIds) {
        foreach ($lineItemIds as $lineItemId) {
            if ($lineItemId == null) {
                continue;
            }


            if ($this->getLineItem($lineItemId) == null) {
                continue;
            }
            $this->_imageLineItemFactory->deleteLineItem($this->getId(), $lineItemId);

        }

        $this->_updateLineItemContainer();

    }

    /**
     * @param $lineitemid
     *
     * @return AbstractImagelineitem
     */
    public function getLineItem($lineitemid)
    {
        if (isset($this->_lineitems[$lineitemid])) {
            return $this->_lineitems[$lineitemid];
        } else {
            return null;
        }
    }

    /**
     * Updates the cart object stucture from the database
     */
    public function _updateLineItemContainer()
    {
        $this->_loadLineItems();
        $this->_loadServiceLineItems();

    }

    /**
     * @return Address
     */

    public function getBillingAddress()
    {
        /**
         * @var AddressFactory $addressFactory
         */
        $addressFactory = AddressFactory::getInstance();

        if ($this->_billingaddress == null && $this->_lineitemcontainer->billingaddressid > 0) {
            $this->_billingaddress = $addressFactory->getAddressById($this->_lineitemcontainer->billingaddressid);
        }
        return $this->_billingaddress;
    }

    /**
     * @return string
     */
    public function getEMail()
    {
        return $this->_lineitemcontainer->email;
    }

    /**
     * @return int returns the current number of line items in this cart
     */
    function getLineItemsCount()
    {
        return count($this->_lineitems);
    }

    /**
     * @return int the sum of all quantities in this cart
     */
    function getLineItemsTotalCount()
    {
        $count = 0;
        /* @var Lineitem $lineitem */
        foreach ($this->getLineItems() as $lineitem) {

            $count += $lineitem->getQuantity();
        }
        return $count;
    }

    /**
     * @return AbstractImagelineitem[] all lineitems from this container
     */
    function getLineItems()
    {
        return array_values($this->_lineitems);
    }

    /**
     * Returns an array of lineitems with the given ImageType
     *
     * @param $imageType ImageType the imageType you want to get the lineitems for.
     * @return array
     */
    function getLineItemsByImageType($imageType) {

        if (isset($this->_lineitemImageTypeCache[$imageType->getId()])) {
            return $this->_lineitemImageTypeCache[$imageType->getId()];
        }

        return array();
    }

    /**
     * returns an array containing the ImageTypes which are in use.
     *
     * @return array
     */
    function getUsedImageTypes() {
        return $this->_usedImageTypesCache;
    }

    /**
     * @return string
     */
    public function getMessage()
    {
        return $this->_lineitemcontainer->message;
    }


    /**
     * @return ServiceLineitem|null
     */
    public function getPaymentMethodServiceLineItem() {
        foreach ($this->_servicelineitems as $servicelineitem) {
            /**
             * @var ServiceLineitem $servicelineitem
             */
            if ($servicelineitem->isPaymentMethod()) {
                return $servicelineitem;
            }
        }

        return null;
    }
    /**
     * @return PaymentMethod|null
     */
    public function getPaymentMethod()
    {
        $sli = $this->getPaymentMethodServiceLineItem();
        if ($sli) {
            return $sli->getMethod();
        }
        return null;
    }

    /**
     * @return string
     */
    public function getPhone()
    {
        return $this->_lineitemcontainer->phone;
    }

    /**
     * @return Address
     */
    public function getShippingAddress()
    {
        /**
         * @var AddressFactory $addressFactory
         */
        $addressFactory = AddressFactory::getInstance();

        if ($this->_shippingaddress == null && $this->_lineitemcontainer->shippingaddressid > 0) {
            $this->_shippingaddress = $addressFactory->getAddressById($this->_lineitemcontainer->shippingaddressid);
        }
        return $this->_shippingaddress;
    }

    /**
     * @return ServiceLineitem|null
     */
    public function getShippingMethodServiceLineItem() {
        foreach ($this->_servicelineitems as $servicelineitem) {
            /**
             * @var ServiceLineitem $servicelineitem
             */
            if ($servicelineitem->isShippingMethod()) {
                return $servicelineitem;
            }
        }

        return null;
    }
    /**
     * @return ShippingMethod|null
     */
    public function getShippingMethod()
    {
        $sli = $this->getShippingMethodServiceLineItem();
        if ($sli) {
            return $sli->getMethod();
        }
        return null;
    }

    /**
     * @return Money
     */
    public function getSubTotal()
    {
        return new Money($this->_lineitemcontainer->subtotal, $this->_lineitemcontainer->subtotalcurrency);
    }


    /**
     * @return ServiceLineitem|null
     */
    public function getSurchargeServiceLineItem() {
        foreach ($this->_servicelineitems as $servicelineitem) {
            /**
             * @var ServiceLineitem $servicelineitem
             */
            if ($servicelineitem->isSurcharge()) {
                return $servicelineitem;
            }
        }

        return null;
    }
    /**
     * @return SurchargeMethod|null
     */
    public function getSurcharge()
    {
        $sli = $this->getSurchargeServiceLineItem();
        if ($sli) {
            return $sli->getMethod();
        }
        return null;
    }

    /**
     * sets a surcharge
     *
     * @param SurchargeMethod $surcharge
     */
    public function setSurcharge($surcharge)
    {

        $this->_deleteMethodByType(ServiceLineitem::TYPE_SURCHARGE);

        if ($surcharge == null) {
            return;
        }

        /* @var ServiceLineitemFactory $serviceLineItemFactory */
        $serviceLineItemFactory = ServiceLineitemFactory::getInstance();
        $serviceLineItemFactory->createLineitem($this, $surcharge);
        $this->_loadServiceLineItems();
    }

    /**
     * @param int $methodtypeid
     */
    protected function _deleteMethodByType($methodtypeid)
    {
        $this->_serviceLineItemFactory->deleteMethodTypeFromLineitemContainer($this->getId(), $methodtypeid);
        $this->_loadServiceLineItems();
    }

    /**
     * @return string
     */
    public function getId()
    {
        return $this->_lineitemcontainer->id;
    }

    /**
     * returns Money the tax amount
     */
    public function getTax() {
        $tax = 0;
        /**
         * @var Lineitem $lineitem
         */
        foreach($this->getLineItems() as $lineitem) {
            /**
             * @var Lineitem $lineitem
             */
            $tax += $lineitem->getTax()->getAmount();
        }

        if ($this->getShippingMethodServiceLineItem()) {
            $tax += $this->getShippingMethodServiceLineItem()->getTax()->getAmount();
        }

        if ($this->getPaymentMethodServiceLineItem()) {
            $tax += $this->getPaymentMethodServiceLineItem()->getTax()->getAmount();
        }

        if ($this->getSurchargeServiceLineItem()) {
            $tax += $this->getSurchargeServiceLineItem()->getTax()->getAmount();
        }

        return new Money($tax, $this->_lineitemcontainer->subtotalcurrency);
    }

    /**
     * @return Money
     */
    public function getTotal()
    {
        return new Money($this->_lineitemcontainer->total, $this->_lineitemcontainer->totalcurrency);
    }

    /**
     * @return string
     */
    public function getUserId()
    {
        return $this->_lineitemcontainer->userid;
    }

    /**
     * @return \Joomla\CMS\User\User|null
     */
    public function getUser() {
        $userid = $this->getUserId();
        if (null == $userid) {
            return null;
        }

        return \Joomla\CMS\Factory::getUser($userid);
    }

    /**
     * @param Address $billingAddress
     */
    public function setBillingAddress($billingAddress)
    {
        $this->_billingaddress = null;
        $this->_lineitemcontainer->billingaddressid = $billingAddress!=null?$billingAddress->getId():-1;
        $this->_storeLineItemContainer();
    }

    protected function _storeLineItemContainer()
    {
        $data = $this->_lineitemcontainer;
        $result = $this->store((array)$data, $this->_lineitemcontainer_table);

        // set the new version number to avoid trouble while saving this item the next time.
        if (isset($result->version) && isset($this->_lineitemcontainer->version)) {
            $this->_lineitemcontainer->version = $result->version;
        }
    }

    /**
     * @param string $email
     */
    public function setEMail($email)
    {
        $this->_lineitemcontainer->email = $email;
        $this->_storeLineItemContainer();
    }

    /**
     * @param string $message
     */
    public function setMessage($message)
    {
        $this->_lineitemcontainer->message = $message;
        $this->_storeLineItemContainer();
    }

    /**
     * sets a Payment
     *
     * @param PaymentMethod $payment
     */
    public function setPaymentMethod($paymentmethod)
    {

        // the payment method contains some data so we should not drop it if we want to use the same method again.
        if ($paymentmethod != null && $this->getPaymentMethod() != null && $paymentmethod->getId() == $this->getPaymentMethod()->getId()) {
            return;
        }

        $this->_deleteMethodByType(ServiceLineitem::TYPE_PAYMENTMETHOD);

        if ($paymentmethod == null) {
            return;
        }

        /* @var ServiceLineitemFactory $serviceLineItemFactory */
        $serviceLineItemFactory = ServiceLineitemFactory::getInstance();
        $serviceLineItemFactory->createLineitem($this,  $paymentmethod);
        $this->_loadServiceLineItems();
    }

    /**
     * @param string $phone
     */
    public function setPhone($phone)
    {
        $this->_lineitemcontainer->phone = $phone;
        $this->_storeLineItemContainer();
    }

    /**
     * @param Address $shippingAddress
     */
    public function setShippingAddress($shippingAddress)
    {
        $this->_shippingaddress = $shippingAddress;
        $this->_lineitemcontainer->shippingaddressid = $shippingAddress!=null?$shippingAddress->getId():-1;
        $this->_storeLineItemContainer();
    }

    /**
     * @param $documentNumber
     */
    public function setDocumentNumber($documentNumber)
    {
        $this->_lineitemcontainer->documentno = $documentNumber;
        $this->_storeLineItemContainer();
    }

    /**
     * sets a shipping
     *
     * @param ShippingMethod $shipping
     */
    public function setShippingMethod($shipping)
    {

        $this->_deleteMethodByType(ServiceLineitem::TYPE_SHIPINGMETHOD);

        if ($shipping == null) {
            return;
        }

        /* @var ServiceLineitemFactory $serviceLineItemFactory */
        $serviceLineItemFactory = ServiceLineitemFactory::getInstance();
        $serviceLineItemFactory->createLineitem($this, $shipping);
        $this->_loadServiceLineItems();
    }

    /**
     * @param Money $price
     */
    public function setSubTotal($price)
    {

        $this->_lineitemcontainer->subtotal = $price->getAmount();
        $this->_lineitemcontainer->subtotalcurrency = $price->getCurrency();
        $this->_storeLineItemContainer();
    }


    /**
     * @param Money $price
     */
    public function setTotal($price)
    {
        $this->_lineitemcontainer->total = $price->getAmount();
        $this->_lineitemcontainer->totalcurrency = $price->getCurrency();
        $this->_storeLineItemContainer();
    }

    /**
     * loads lineitems from the database
     *
     */
    protected function _loadLineItems()
    {
        $this->_lineitemImageTypeCache = array();
        $this->_usedImageTypesCache = array();

        /**
         * @var ImageLineitemFactory $imageLineItemFactory
         * @var AbstractImagelineitem $lineitem
         */
        $imageLineItemFactory = ImageLineitemFactory::getInstance();

        $lineitems = $imageLineItemFactory->getLineItemsByLineItemContainerId($this->getId());

        $indexedLineitems = array();
        $this->_lineitemImageTypeCache = array();
        foreach ($lineitems as $lineitem) {
            $indexedLineitems[$lineitem->getId()] = $lineitem;

            $imagetype = $lineitem->getImageType();
            if (null != $imagetype) {
                if (!isset($this->_lineitemImageTypeCache[$imagetype->getId()])) {
                    $this->_lineitemImageTypeCache[$imagetype->getId()] = array();
                    array_push($this->_usedImageTypesCache, $imagetype);
                }
                array_push($this->_lineitemImageTypeCache[$imagetype->getId()], $lineitem);
            }

        }

        $this->_lineitems = $indexedLineitems;
    }

    /**
     */
    protected function _loadServiceLineItems()
    {
        /**
         * @var ServiceLineitemFactory $serviceLineItemFactory
         * @var ServiceLineitem $lineitem
         */
        $serviceLineItemFactory = ServiceLineitemFactory::getInstance();

        $lineitems = $serviceLineItemFactory->getLineItemsByLineItemContainerId($this->getId());

        $indexedLineitems = array();
        foreach ($lineitems as $lineitem) {
            $indexedLineitems[$lineitem->getId()] = $lineitem;
        }

        $this->_servicelineitems = $indexedLineitems;
    }

    /**
     * @return array|null
     */
    public function getServiceLineItems() {
        return $this->_servicelineitems;
    }

    public function getCreationDate() {
        return $this->_lineitemcontainer->created;
    }

    public function getModificationDate() {
        return $this->_lineitemcontainer->modified;
    }

    public function getDocumentNumber() {
        return $this->_lineitemcontainer->documentno;
    }

    public function getType() {
        $digital = false;
        $physical = false;
        foreach($this->getLineItems() as $imagelineitem) {
            /**
             * @var AbstractImagelineitem $imagelineitem
             */
            if ($imagelineitem->getImageType()->isDigital()) {
                $digital = true;
            } else {
                $physical = true;
            }
        }

        if ($digital && $physical) {
            return BasketType::TYPE_MIXED;
        }

        if ($digital) {
            return BasketType::TYPE_DIGITAL;
        }

        return BasketType::TYPE_PHYSICAL;
    }

    public function getFirstname() {
        return $this->_lineitemcontainer->firstname;
    }

    public function getLastname() {
        return $this->_lineitemcontainer->lastname;
    }

    /**
     * @param string $email
     */
    public function setFirstname($firstname)
    {
        $this->_lineitemcontainer->firstname = $firstname;
        $this->_storeLineItemContainer();
    }

    /**
     * @param string $email
     */
    public function setLastname($lastname)
    {
        $this->_lineitemcontainer->lastname = $lastname;
        $this->_storeLineItemContainer();
    }


}
