<?php

/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Svenbluege\Component\Eventgallery\Site\Library\Manager;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Factory;
use Joomla\CMS\Mail\MailHelper;
use Joomla\CMS\User\User;
use Smarty;
use Svenbluege\Component\Eventgallery\Site\Library\Common\GeoObjects;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\EmailTemplateFactory;
use Svenbluege\Component\Eventgallery\Site\Library\ImageLineitem;
use Svenbluege\Component\Eventgallery\Site\Library\Order;

defined('_JEXEC') or die();

class EmailTemplateManager extends  AbstractManager
{

    /** Holds the available keys for now*/
    private $_values = array(
            // mail for new order events
            'new_order'     => 'COM_EVENTGALLERY_EMAILTEMPLATE_EMAIL_NEW_ORDER',
            // mail for paid order events.
            'paid_order'    => 'COM_EVENTGALLERY_EMAILTEMPLATE_EMAIL_PAID_ORDER',
            // mail for shipped order events.
            'shipped_order'  => 'COM_EVENTGALLERY_EMAILTEMPLATE_EMAIL_SHIPPED_ORDER'
        );

    /**
    * return a hash which contains the key and the translation key for an email template key.
    */
    public function getEmailtemplateKeys() {
        return $this->_values;
    }


    /**
     * returns the display name as a localization key for a given key.
     * @param $key
     * @return string
     */
    public function getEmailtemplateKeyDisplayName($key) {

        if (isset($this->_values[$key])) {
            return $this->_values[$key];
        }

        return "";
    }

    /**
     * Send out a mail based on aa email template. Will replace the placeholders in the template as well based on the given data.
     *
     * @param $key
     * @param $language
     * @param $publishedOnly
     * @param $data array Array which holds the data structure
     * @param $to array|string the receiver of this mail, array is email, name
     * @param $sendCopyToAdmins boolean defines if we send the same mail to the admin
     * @return mixed|string
     */
    public function sendMail($key, $language, $publishedOnly, $data, $to, $sendCopyToAdmins) {
        /**
         * @var EmailTemplateFactory $emailtemplateFactory
         */
        $emailtemplateFactory = EmailTemplateFactory::getInstance();
        $emailtemplate = $emailtemplateFactory->getEmailtemplateByKey($key, $language, $publishedOnly);

        if (null == $emailtemplate) {

            $subject = '';
            $body = '';
            $attachments = Array();

        } else {
            $subject = $emailtemplate->getSubject();
            $body = $emailtemplate->getBody();
            $attachments = $emailtemplate->getAttachments();
        }

        if (empty(trim($subject))) {
            $subject = $this->getDefaultSubject($key);
        }

        if (empty(trim($body))) {
            $body = $this->getDefaultBody($key);
        }


        return $this->sendMailBySubjectAndBody($subject, $body, $data, $attachments, $to, $sendCopyToAdmins);
    }

    /**
     * Send out a mail based on aa email template. Will replace the placeholders in the template as well based on the given data.
     *
     * @param int $id
     * @param $data array Array which holds the data structure
     * @param $to array|string the receiver of this mail, array is email, name
     * @param $sendCopyToAdmins boolean defines if we send the same mail to the admin
     * @internal param int $id ID of the email template
     * @return boolean
     */
    public function sendMailById($id, $data, $to, $sendCopyToAdmins = true)
    {
        /**
         * @var EmailTemplateFactory $emailtemplateFactory
         */
        $emailtemplateFactory = EmailTemplateFactory::getInstance();
        $emailtemplate = $emailtemplateFactory->getEmailtemplateById($id);

        $subject = $emailtemplate->getSubject();
        $body = $emailtemplate->getBody();
        $attachments = $emailtemplate->getAttachments();

        if (empty(trim($subject))) {
            $subject = $this->getDefaultSubject($emailtemplate->getKey());
        }

        if (empty(trim($body))) {
            $body = $this->getDefaultBody($emailtemplate->getKey());
        }

         return $this->sendMailBySubjectAndBody($subject, $body, $data, $attachments, $to, $sendCopyToAdmins);
    }

    /**
     * sends a mail using the given data.
     *
     * @param $subject
     * @param $body
     * @param $data
     * @param $to
     * @param $sendCopyToAdmins
     * @return mixed|string
     */
    public function sendMailBySubjectAndBody($subject, $body, $data, $attachments, $to, $sendCopyToAdmins) {
        /**
         * @var Joomla\Registry\Registry $config
         * @var Joomla\Registry\Registry $params
         */
        $config = Factory::getConfig();
        $mailer = Factory::getMailer();
        $egConfig = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();

        $subject = $this->populate($subject, $data);
        $body = $this->populate($body, $data);

        $mailer->setSubject(
            $subject
        );

        $mailer->isHtml(true);
        $mailer->Encoding = 'base64';
        $mailer->setBody($body);

        foreach($attachments as $attachment) {
            $filepath = JPATH_ROOT . '/images/' . $attachment;
            if (file_exists($filepath)) {
                $mailer->addAttachment($filepath);
            }
        }

        // Customer Mail
        $sender = array(
            $config->get( 'mailfrom' ),
            $config->get( 'fromname' ) );

        $mailer->setSender($sender);

        if (is_array($to) && count($to)==2) {
            $mailer->addRecipient($to[0], $to[1]);
        } else {
            $mailer->addRecipient($to);
        }

        $send = $mailer->Send();

        if ($send !== true) {
            return $mailer->ErrorInfo;
        }

        // Admin Mail
        if (!$sendCopyToAdmins) {
            return true;
        }

        $mailer->clearReplyTos();
        $mailer->clearAllRecipients();
        $mailer->clearAddresses();
        $mailer->clearBCCs();
        $mailer->clearCCs();

        if (is_array($to) && count($to)==2) {
            $mailer->addReplyTo($to[0], $to[1]);
        } else {
            $mailer->addReplyTo($to);
        }

        $userids = Access::getUsersByGroup($egConfig->getGeneral()->getAdminUserGroupId());

        foreach ($userids as $userid) {
            $user = User::getInstance($userid);
            if ($user->sendEmail==1) {
                $mailadresse = MailHelper::cleanAddress($user->email);
                $mailer->addRecipient($mailadresse);
            }
        }

        $send = $mailer->Send();

        if ($send !== true) {
            return $mailer->ErrorInfo;
        }

        return $send;

    }

    /**
     * Renders the placeholders in a string with the given data.
     *
     * @param $text string the input string
     * @param $data array an array containing the data
     * @return string
     */
    public function populate($text, $data) {
        $smarty = new Smarty();

        $smarty->debugging = false;
        $smarty->caching = false;
        $smarty->setCacheDir(JPATH_ROOT.DIRECTORY_SEPARATOR.'cache/com_eventgallery_template_cache');
        $smarty->setCompileDir(JPATH_ROOT.DIRECTORY_SEPARATOR.'cache/com_eventgallery_template_compile');

        $smarty->assign("data", $data);

        $renderedText = $smarty->fetch('string:' . $text);

        return $renderedText;
    }



    /**
     * @param $key string the mail template key
     * @returns string the subject template
     */
    public function getDefaultSubject($key) {
        return $this->getTemplateFile($key, 'subject');
    }

    /**
     * @param $key string the mail template key
     * @returns string the subject template
     */
    public function getDefaultBody($key) {
        return $this->getTemplateFile($key, 'body');
    }

    /**
     * @param $key string the mail template key
     * @return array the demo data array
     */
    public function getDemoData($key) {
        return json_decode($this->getTemplateFile($key, 'demodata', 'json'));
    }

    /**
     * @param $key string the mail template key
     * @param $type string the type of the template: subject|body
     * @return string
     */
    private function getTemplateFile($key, $type, $ext = 'tpl') {
        $path = JPATH_ROOT.'/components/com_eventgallery/tmpl/mail/'.$key.'_'.$type.'.'.$ext;

        if (file_exists($path)) {
            return file_get_contents($path);
        }

        return "";
    }

    /**
     * Transforms the order object into an array holding data we can use in a email template
     *
     * @param $order Order
     * @return array an array containing the order data
     * @throws Exception
     */
    public function createOrderData($order, $show_vat) {

        $orderData = Array(
            "firstname" => $order->getFirstname(),
            "lastname" => $order->getLastname(),
            "message" => $order->getMessage(),
            "phone" => $order->getPhone(),
            "email" => $order->getEMail(),
            "documentnumber" => $order->getDocumentNumber(),
            "date" => $order->getCreationDate(),
            "lineitemscount" => $order->getLineItemsCount(),
            "lineitemstotalcount" => $order->getLineItemsTotalCount(),
            "subtotal" => (string)$order->getSubTotal(),
            "total" => (string)$order->getTotal(),
            "newsletter_optin" => $order->isNewsletterOptIn() ? 1 : 0
        );

        if ($show_vat) {
            $orderData['tax'] = (string)$order->getTax();
        }

        if ($order->getBillingAddress() != null) {
            $orderData['billingaddress'] = Array(
                "firstname" => $order->getBillingAddress()->getFirstName(),
                "lastname" => $order->getBillingAddress()->getLastName(),
                "address1" => $order->getBillingAddress()->getAddress1(),
                "address2" => $order->getBillingAddress()->getAddress2(),
                "address3" => $order->getBillingAddress()->getAddress3(),
                "zip" => $order->getBillingAddress()->getZip(),
                "city" => $order->getBillingAddress()->getCity(),
                "state" => GeoObjects::getStateName($order->getBillingAddress()->getState()),
                "stateid" => GeoObjects::getStateID($order->getBillingAddress()->getState()),
                "country" => GeoObjects::getCountryName($order->getBillingAddress()->getCountry())
            );
        }

        if ($order->getShippingAddress() != null) {
            $orderData['shippingaddress'] = array(
                "firstname" => $order->getShippingAddress()->getFirstName(),
                "lastname" => $order->getShippingAddress()->getLastName(),
                "address1" => $order->getShippingAddress()->getAddress1(),
                "address2" => $order->getShippingAddress()->getAddress2(),
                "address3" => $order->getShippingAddress()->getAddress3(),
                "zip" => $order->getShippingAddress()->getZip(),
                "city" => $order->getShippingAddress()->getCity(),
                "state" => GeoObjects::getStateName($order->getShippingAddress()->getState()),
                "stateid" => GeoObjects::getStateID($order->getShippingAddress()->getState()),
                "country" => GeoObjects::getCountryName($order->getShippingAddress()->getCountry())
            );
        }


        if (null != $order->getSurchargeServiceLineItem()) {
            $orderData['surcharge'] = Array(
                "name" => $order->getSurchargeServiceLineItem()->getDisplayName(),
                "description" => $order->getSurchargeServiceLineItem()->getDescription(),
                "price" => (string)$order->getSurchargeServiceLineItem()->getPrice(),
                "content" => $order->getSurchargeServiceLineItem()->getMethod()->getMethodConfirmContent($order, true)
            );
        }

        if (null != $order->getShippingMethodServiceLineItem()) {
            $orderData['shipping'] = Array(
                "name" => $order->getShippingMethodServiceLineItem()->getDisplayName(),
                "description" => $order->getShippingMethodServiceLineItem()->getDescription(),
                "price" => (string)$order->getShippingMethodServiceLineItem()->getPrice(),
                "content" => $order->getShippingMethodServiceLineItem()->getMethod()->getMethodConfirmContent($order, true)
            );
        }

        if (null != $order->getPaymentMethod()) {
            $orderData['payment'] = Array(
                "name" => $order->getPaymentMethodServiceLineItem()->getDisplayName(),
                "description" => $order->getPaymentMethodServiceLineItem()->getDescription(),
                "price" => (string)$order->getPaymentMethodServiceLineItem()->getPrice(),
                "content" => $order->getPaymentMethodServiceLineItem()->getMethod()->getMethodConfirmContent($order, true)
            );
        }

        $lineitems = Array();
        /**
         * @var ImageLineitem $lineitem
         */
        foreach($order->getLineItems() as $lineitem) {
            $lineitemData = Array(
                "quantity"  => $lineitem->getQuantity(),
                "price" => (string)$lineitem->getPrice(),
                "priceincluded" => (string)$lineitem->isPriceIncluded()?1:0,
                "singleprice" => (string)$lineitem->getSinglePrice(),
                "foldername" => $lineitem->getFolderName(),
                "filename" => $lineitem->getFileName(),
                "original_filename" => $lineitem->getOriginalFilename(),
                "original_filetitle" => $lineitem->getFileTitle(),
                "original_filecaption" => $lineitem->getFileCaption(),
                "buyernote" => $lineitem->getBuyerNote(),
                "imagetype" => $lineitem->getImageType()==null?'' : Array("name"  => $lineitem->getImageType()->getDisplayName()),
                "imageurl" =>  $lineitem->getFile()==null?'' : $lineitem->getFile()->getImageUrl(NULL, NULL, true),
                "thumburl" => $lineitem->getMailThumbUrl()
        );
            array_push($lineitems, $lineitemData);
        }

        $orderData['imagelineitems'] = $lineitems;

        return $orderData;
    }

}
