<?php
/**
 * @package     Sven.Bluege
 * @subpackage  com_eventgallery
 *
 * @copyright   Copyright (C) 2005 - 2019 Sven Bluege All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
namespace Svenbluege\Component\Eventgallery\Site\Model;
use Joomla\CMS\Factory;
use Joomla\CMS\Pagination\Pagination;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\Query\QueryElement;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FileFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\FolderFactory;
use Svenbluege\Component\Eventgallery\Site\Library\Factory\TagsFactory;
use Svenbluege\Component\Eventgallery\Site\Library\File\File;
use Svenbluege\Component\Eventgallery\Site\Library\Folder\Folder;
use Svenbluege\Component\Eventgallery\Site\Library\Helper\Tags;
defined('_JEXEC') or die();

class EventsModel extends \Joomla\CMS\MVC\Model\BaseDatabaseModel
{

    var $_total = 0;
    var $_entries = null;
    var $_pagination;

    private $config;
    /**
     * @var JCacheControllerCallback $cache
     */
    protected $cache;

    function __construct()
    {
        parent::__construct();

        $this->config = \Svenbluege\Component\Eventgallery\Site\Library\Configuration\Main::getInstance();


        $this->cache = Factory::getCache('com_eventgallery');
        if ($this->config->getGeneral()->doUseCaching()) {
            $this->cache->setCaching(true);
        }

        $app = Factory::getApplication();

        $limitstart =  $app->input->getInt('limitstart');
        $limit =  $this->config->getEventsList()->getMaxEventsPerPage();
        $this->setState('limit',$limit);
        $this->setState('com_eventgallery.events.limitstart',$limitstart);
    }

    /**
     * This method gets the entries for this model. It uses caching to prevent getting data multiple times.
     *
     * @param boolean $isForNavigation This method can be used to run a search or do do navigation. Doing navigation will result in fields which are populated with _total and _entries. In addition limit and limitstart are taken into account from the state.
     * @param int $limitstart
     * @param int $limit
     * @param array $tags
     * @param string $sortAttribute
     * @param $usergroups
     * @param int $catid the category id to filter the events
     * @param bool $recursive defines if we should get the events for the subcategories too.
     * @param bool $filterByUserGroups defines if we remove user group specific events from list which are invisible for the current user
     * @return array
     */
    function getEntries($isForNavigation, $limitstart=0, $limit=0, $tags = array(), $sortAttribute='ordering', $usergroups = null, $catid = null, $recursive = false, $filterByUserGroups=true)
    {
        if ($isForNavigation) {

            if ($limit == 0) {
                $limit = $this->getState('limit');
            } else {
                $this->setState('limit', $limit);

            }

            if ($limitstart == 0) {
                $limitstart = $this->getState('com_eventgallery.events.limitstart');
            }
        }

        // fix issue with events list where paging was working
        if($limitstart == null || $limitstart <0 ) {
            $limitstart = 0;
        }

        $entries =  $this->cache->get(
            array($this, 'getEntriesUnCached'), array($limitstart, $limit, $tags, $sortAttribute, $usergroups, $catid, $recursive, $filterByUserGroups)
        );

        if ($isForNavigation) {
            $this->_entries = $entries;
            $this->_total = count($entries);
        }
        /*
         * Avoid removing one item from the result if limit is set to -1
         */
        return array_slice($entries, $limitstart, $limit<=0 ? null : $limit);
    }

    function getImages($limit, $tags, $sortAttribute , $usergroups = null, $catid = null, $recursive = false, $filterByUserGroups=true)
    {
        $files =  $this->cache->get(
            array($this, 'getImagesUnCached'), array($limit, $tags, $sortAttribute, $usergroups, $catid, $recursive, $filterByUserGroups)
        );

        return $files;
    }

    /**
     * just get the entries for this model.
     *
     * @param int $limitstart
     * @param int $limit
     * @param array $tags
     * @param string $sortAttribute
     * @param array $usergroups array even if unused we need this for the cache call
     * @param int $catid the category id to filter the events
     * @param bool $recursive defines if we should get the events for the subcategories too.
     * @param bool $filterByUserGroups defines if we remove user group specific events from list which are invisible for the current user
     * @return array
     */
    function getEntriesUnCached(/** @noinspection PhpUnusedParameterInspection */$limitstart=0, $limit=0, $tags = array(), $sortAttribute='ordering', $usergroups = null, $catid = null, $recursive = false, $filterByUserGroups = true)
    {

        $query = $this->_db->getQuery(true)
            ->select('folder.*, count(1) AS '.$this->_db->quoteName('overallCount'))
            ->from($this->_db->quoteName('#__eventgallery_folder') . ' AS folder')
            ->join('', $this->_db->quoteName('#__eventgallery_file') . ' AS file ON folder.folder = file.folder and file.published=1')
            ->where('(file.ismainimageonly IS NULL OR file.ismainimageonly=0)')
            ->where('folder.published=1')
            ->group('folder.id');

        if (null != $catid) {
            $catIds = $this->getCategoryIds((int)$catid, $recursive);
            $query->where('catid in ('. implode(',', $catIds) .') ');
        }

        // Define null and now dates
        $nullDate = $this->_db->quote($this->_db->getNullDate());
        $nowDate  = $this->_db->quote(Factory::getDate()->toSql());

        $query->where('(folder.publish_up is null OR folder.publish_up = ' . $nullDate . ' OR folder.publish_up <= ' . $nowDate . ')')
              ->where('(folder.publish_down is null OR folder.publish_down = ' . $nullDate . ' OR folder.publish_down >= ' . $nowDate . ')');

        if ($sortAttribute == "date_asc") {
            $query->order('date ASC, ordering DESC');
        } elseif ($sortAttribute == "date_desc") {
            $query->order('date DESC, ordering DESC');
        } elseif ($sortAttribute == "name_asc") {
            $query->order('folder.folder ASC');
        } elseif ($sortAttribute == "name_desc") {
            $query->order('folder.folder DESC');
        } elseif ($sortAttribute == "hits_asc") {
            $query->order('folder.hits ASC');
        } elseif ($sortAttribute == "hits_desc") {
            $query->order('folder.hits DESC');
        } elseif ($sortAttribute == "random") {
            $query->order('RAND()');
        }else {
            $query->order('ordering DESC');
        }

        $entries = $this->_getList($query);


        $ids = array_map(function($entry) {
            return $entry->id;
        }, $entries);

        /**
         * @var TagsFactory $tagsFactory
         */
        $tagsFactory = TagsFactory::getInstance();
        $tagsFactory->prefillCache($ids);

        /**
         * @var FolderFactory $folderFactory
         */
        $folderFactory = FolderFactory::getInstance();
        $filteredfolders = [];
        foreach ($entries as $entry)
        {
            $folder = $folderFactory->getFolder($entry->folder);
            if ($folder->isVisible() && $folder->checkTags($tags)) {
                array_push($filteredfolders, $folder);
            }

        }

        return $filteredfolders;

    }

    function getImagesUnCached($limit=0, $tags = array(), $sortAttribute='ordering', $usergroups = null, $catid = null, $recursive = false, $filterByUserGroups = true)
    {
        /**
         * @var DatabaseQuery $query
         */
        $query = $this->_db->getQuery(true);
        $query->select('folder.*, file.file, file.creation_date')
            ->from($this->_db->quoteName('#__eventgallery_folder') . ' AS folder')
            ->join('', $this->_db->quoteName('#__eventgallery_file') . ' AS file ON folder.folder = file.folder and file.published=1')
            ->where('(file.ismainimageonly IS NULL OR file.ismainimageonly=0)')
            ->where('folder.published=1')
            ->setLimit($limit*2);



        if (null != $catid) {
            $catIds = $this->getCategoryIds((int)$catid, $recursive);
            $query->where('catid in ('. implode(',', $catIds) .') ');
        }

        // Define null and now dates
        $nullDate = $this->_db->quote($this->_db->getNullDate());
        $nowDate  = $this->_db->quote(Factory::getDate()->toSql());

        $query->where('(folder.publish_up is null OR folder.publish_up = ' . $nullDate . ' OR folder.publish_up <= ' . $nowDate . ')')
            ->where('(folder.publish_down is null OR folder.publish_down = ' . $nullDate . ' OR folder.publish_down >= ' . $nowDate . ')');

        if ($sortAttribute == "date_asc") {
            $query->order('file.creation_date ASC, folder.date ASC, ordering DESC');
            $query->where('file.creation_date is not null and file.creation_date != ""');
        } elseif ($sortAttribute == "date_desc") {
            $query->order('file.creation_date DESC, folder.date DESC, ordering DESC');
            $query->where('file.creation_date is not null and file.creation_date != ""');
        } elseif ($sortAttribute == "name_asc") {
            $query->order('file.file ASC');
        } elseif ($sortAttribute == "name_desc") {
            $query->order('file.file DESC');
        } elseif ($sortAttribute == "random") {
            $query->order('RAND()');
        }



        $queryResult = $this->_getList($query);


        $files = Array();
        /**
         * @var FileFactory $fileFactory
         */
        $fileFactory = FileFactory::getInstance();

        foreach ($queryResult as $rawFile)
        {

            $file = $fileFactory->getFile($rawFile->folder, $rawFile->file);
            $folder = $file->getFolder();
            if ($folder->isPublicVisible() && $file->isPublished() && $folder->checkTags($tags)) {
                array_push($files, $file);
            }
            if (count($files) >= $limit) {
                break;
            }
        }

        return $files;

    }

    /**
     * returns the paging bar for the current data set.
     *
     * @return Pagination
     */
    function getPagination($catid)
    {

        $app = Factory::getApplication();

        if (empty($this->_pagination))
        {

            $total = $this->_total;

            /**
             * @var integer $limit
             */
            $limit      = $this->getState('limit');

            /**
             * @var integer $limitstart
             */
            $limitstart = $this->getState('com_eventgallery.events.limitstart');


            if ($limitstart > $total || $app->input->getInt('limitstart','0')==0) {
                $limitstart=0;
                $this->setState('com_eventgallery.event.limitstart',$limitstart);
            }

            $this->_pagination = new Pagination($total, $limitstart, $limit);
            if (!empty($catid) && $catid != 'root') {
                $this->_pagination->setAdditionalUrlParam('catid', $catid);
            }

        }

        return $this->_pagination;

    }

    /**
     * @param int $catid
     * @param bool $recursive
     * @return int[]
     */
    private function getCategoryIds(int $catid, bool $recursive): array
    {
        $catIds = array((int)$catid);

        if ($recursive) {
            $options = array();
            $categories = \Joomla\CMS\Categories\Categories::getInstance('Eventgallery', $options);

            /**
             * @var JCategoryNode $currentCategory
             * @var JCategoryNode $childCategory
             */

            $currentCategory = $categories->get($catid);

            if ($currentCategory != null) {
                $childCategories = $currentCategory->getChildren(true);
                foreach ($childCategories as $childCategory) {
                    array_push($catIds, (int)$childCategory->id);
                }
            }

        }
        return $catIds;
    }

}
